<?php
/**
 * Plugin Name: SafetyPress - WordPress & WooCommerce Security
 * Plugin URI: https://safetypress.de
 * Description: Das umfassendste Sicherheits-Plugin für WordPress und WooCommerce. Firewall, Login-Schutz, Malware-Scanner, 2FA und mehr.
 * Version: 1.2.1
 * Author: SafetyPress
 * Author URI: https://safetypress.de
 * License: GPL v2 or later
 * Text Domain: safetypress
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit;
}

define('SAFETYPRESS_VERSION', '1.2.1');
define('SAFETYPRESS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SAFETYPRESS_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SAFETYPRESS_PLUGIN_BASENAME', plugin_basename(__FILE__));

// ============================================
// AUTO-UPDATER LADEN
// ============================================

require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-updater.php';

final class SafetyPress {

    private static $instance = null;
    
    // Core-Module
    public $firewall = null;
    public $login_protection = null;
    public $hardening = null;
    public $scanner = null;
    public $logger = null;
    public $woo_protection = null;
    public $two_factor = null;
    public $license = null;
    public $stripe = null;
    
    // Neue Module v1.2.0
    public $notifications = null;
    public $scheduled_scanner = null;
    public $login_customizer = null;
    public $integrity_checker = null;
    public $security_score = null;

    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->load_files();
        $this->init_hooks();
    }

    private function load_files() {
        // Core-Klassen
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-helper.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-logger.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-license.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-stripe.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-firewall.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-login-protection.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-hardening.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-scanner.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-two-factor.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-woo-protection.php';
        
        // Neue Klassen v1.2.0
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-notifications.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-scheduled-scanner.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-login-customizer.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-integrity-checker.php';
        require_once SAFETYPRESS_PLUGIN_DIR . 'includes/class-security-score.php';
        
        if (is_admin()) {
            require_once SAFETYPRESS_PLUGIN_DIR . 'admin/class-admin.php';
        }
    }

    private function init_hooks() {
        add_action('plugins_loaded', array($this, 'init_modules'), 1);
        add_action('init', array($this, 'load_textdomain'));
        
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    public function init_modules() {
        // Core-Module
        $this->license = new SafetyPress_License();
        $this->stripe = new SafetyPress_Stripe();
        $this->logger = new SafetyPress_Logger();
        $this->firewall = new SafetyPress_Firewall();
        $this->login_protection = new SafetyPress_Login_Protection();
        $this->hardening = new SafetyPress_Hardening();
        $this->scanner = new SafetyPress_Scanner();
        $this->two_factor = new SafetyPress_Two_Factor();
        
        // Neue Module v1.2.0
        $this->notifications = new SafetyPress_Notifications();
        $this->scheduled_scanner = new SafetyPress_Scheduled_Scanner();
        $this->login_customizer = new SafetyPress_Login_Customizer();
        $this->integrity_checker = new SafetyPress_Integrity_Checker();
        $this->security_score = new SafetyPress_Security_Score();
        
        // WooCommerce Schutz
        if (class_exists('WooCommerce')) {
            $this->woo_protection = new SafetyPress_Woo_Protection();
        }
        
        if (is_admin()) {
            new SafetyPress_Admin();
        }
    }

    public function load_textdomain() {
        load_plugin_textdomain('safetypress', false, dirname(SAFETYPRESS_PLUGIN_BASENAME) . '/languages');
    }

    public function activate() {
        global $wpdb;
        $charset = $wpdb->get_charset_collate();
        
        // Alten Scan-Cache löschen
        delete_transient('safetypress_last_scan');
        
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        
        dbDelta("CREATE TABLE IF NOT EXISTS {$wpdb->prefix}safetypress_logs (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            event_type varchar(50) NOT NULL,
            event_message text NOT NULL,
            ip_address varchar(45) NOT NULL,
            user_id bigint(20) DEFAULT 0,
            user_agent text,
            extra_data longtext,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset;");
        
        dbDelta("CREATE TABLE IF NOT EXISTS {$wpdb->prefix}safetypress_blocked_ips (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            ip_address varchar(45) NOT NULL,
            reason varchar(255) NOT NULL,
            blocked_until datetime DEFAULT NULL,
            is_permanent tinyint(1) DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY ip_address (ip_address)
        ) $charset;");
        
        dbDelta("CREATE TABLE IF NOT EXISTS {$wpdb->prefix}safetypress_login_attempts (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            ip_address varchar(45) NOT NULL,
            username varchar(255) NOT NULL,
            attempt_time datetime DEFAULT CURRENT_TIMESTAMP,
            success tinyint(1) DEFAULT 0,
            PRIMARY KEY (id)
        ) $charset;");
        
        dbDelta("CREATE TABLE IF NOT EXISTS {$wpdb->prefix}safetypress_checksums (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            file_path varchar(500) NOT NULL,
            file_hash varchar(64) NOT NULL,
            file_size bigint(20) NOT NULL,
            last_modified datetime NOT NULL,
            scanned_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset;");
        
        // Standard-Einstellungen
        $defaults = array(
            // Core
            'enable_firewall' => 1,
            'enable_login_protection' => 1,
            'enable_hardening' => 1,
            'max_login_attempts' => 5,
            'lockout_duration' => 30,
            'hide_wp_version' => 1,
            'disable_xmlrpc' => 1,
            'disable_file_editor' => 1,
            'enable_woo_protection' => 1,
            'fraud_score_threshold' => 70,
            'enable_2fa' => 0,
            'enable_activity_log' => 1,
            'log_retention_days' => 30,
            'license_key' => '',
            'license_status' => 'free',
            
            // Neu v1.2.0: E-Mail-Benachrichtigungen
            'email_notifications' => 1,
            'notification_email' => get_option('admin_email'),
            
            // Neu v1.2.0: Geplante Scans
            'scheduled_scan' => 'disabled',
            'scheduled_scan_email' => 1,
            
            // Neu v1.2.0: Login-URL
            'custom_login_url' => '',
            'enable_custom_login' => 0,
            
            // Neu v1.2.0: Security Headers
            'enable_csp' => 0,
            'csp_mode' => 'basic',
        );
        
        foreach ($defaults as $key => $value) {
            if (get_option('safetypress_' . $key) === false) {
                add_option('safetypress_' . $key, $value);
            }
        }
        
        // Cron-Jobs planen
        if (!wp_next_scheduled('safetypress_cleanup')) {
            wp_schedule_event(time(), 'daily', 'safetypress_cleanup');
        }
        
        // MU-Plugin für Custom Login URL erstellen
        $this->install_mu_plugin();
        
        // .htaccess prüfen und ergänzen falls nötig
        $this->ensure_htaccess();
        
        // Aktivierungs-Log
        $wpdb->insert($wpdb->prefix . 'safetypress_logs', array(
            'event_type' => 'plugin_activated',
            'event_message' => 'SafetyPress v' . SAFETYPRESS_VERSION . ' wurde aktiviert',
            'ip_address' => isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field($_SERVER['REMOTE_ADDR']) : '127.0.0.1',
            'user_id' => get_current_user_id(),
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field($_SERVER['HTTP_USER_AGENT']) : '',
            'created_at' => current_time('mysql'),
        ));
    }
    
    /**
     * MU-Plugin für Custom Login URL installieren
     */
    private function install_mu_plugin() {
        $mu_dir = ABSPATH . 'wp-content/mu-plugins';
        $mu_file = $mu_dir . '/safetypress-login.php';
        
        // Ordner erstellen falls nicht vorhanden
        if (!is_dir($mu_dir)) {
            wp_mkdir_p($mu_dir);
        }
        
        // MU-Plugin Code
        $mu_code = '<?php
/**
 * SafetyPress Login URL Handler
 * Automatisch erstellt von SafetyPress
 */

$request_uri = $_SERVER["REQUEST_URI"] ?? "";

// Debug
if (strpos($request_uri, "sp-debug") !== false) {
    global $wpdb;
    header("Content-Type: text/plain; charset=utf-8");
    echo "SafetyPress MU-Plugin aktiv!\n\n";
    
    if ($wpdb) {
        $custom_slug = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'safetypress_custom_login_url\'");
        $enabled = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'safetypress_enable_custom_login\'");
        $home = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'home\'");
        
        echo "custom_slug: [$custom_slug]\n";
        echo "enabled: [$enabled]\n";
        echo "home: [$home]\n";
        echo "REQUEST_URI: [$request_uri]\n";
        echo "\n=> Login-URL: $home/$custom_slug\n";
    }
    exit;
}

// Login Handler
global $wpdb;
if (!$wpdb) return;

$custom_slug = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'safetypress_custom_login_url\'");
$enabled = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'safetypress_enable_custom_login\'");

if (empty($custom_slug) || !$enabled) return;

$path = parse_url($request_uri, PHP_URL_PATH);
$path = trim($path, "/");

$home_url = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'home\'");
$home_path = parse_url($home_url, PHP_URL_PATH);
$home_path = $home_path ? trim($home_path, "/") : "";

$slug = $path;
if ($home_path && strpos($path, $home_path) === 0) {
    $slug = substr($path, strlen($home_path));
    $slug = ltrim($slug, "/");
}

if ($slug === $custom_slug) {
    setcookie("safetypress_login_access", md5($custom_slug . "secret"), time() + 3600, "/");
    $admin_url = $home_url . "/wp-admin/";
    $login_url = $home_url . "/wp-login.php?redirect_to=" . urlencode($admin_url);
    header("Location: " . $login_url);
    exit;
}
';
        
        // Datei schreiben
        file_put_contents($mu_file, $mu_code);
    }
    
    /**
     * .htaccess prüfen und WordPress-Regeln hinzufügen falls nötig
     */
    private function ensure_htaccess() {
        $htaccess_file = ABSPATH . '.htaccess';
        
        // Wenn keine .htaccess existiert oder WordPress-Regeln fehlen
        if (!file_exists($htaccess_file)) {
            $home_path = parse_url(home_url(), PHP_URL_PATH);
            $home_path = $home_path ? trim($home_path, '/') . '/' : '/';
            
            $htaccess_content = "# BEGIN WordPress\n";
            $htaccess_content .= "<IfModule mod_rewrite.c>\n";
            $htaccess_content .= "RewriteEngine On\n";
            $htaccess_content .= "RewriteBase /{$home_path}\n";
            $htaccess_content .= "RewriteRule ^index\\.php$ - [L]\n";
            $htaccess_content .= "RewriteCond %{REQUEST_FILENAME} !-f\n";
            $htaccess_content .= "RewriteCond %{REQUEST_FILENAME} !-d\n";
            $htaccess_content .= "RewriteRule . /{$home_path}index.php [L]\n";
            $htaccess_content .= "</IfModule>\n";
            $htaccess_content .= "# END WordPress\n";
            
            file_put_contents($htaccess_file, $htaccess_content);
        }
    }

    public function deactivate() {
        wp_clear_scheduled_hook('safetypress_cleanup');
        wp_clear_scheduled_hook('safetypress_daily_scan');
        wp_clear_scheduled_hook('safetypress_weekly_scan');
        wp_clear_scheduled_hook('safetypress_integrity_check');
        
        // MU-Plugin entfernen
        $mu_file = ABSPATH . 'wp-content/mu-plugins/safetypress-login.php';
        if (file_exists($mu_file)) {
            unlink($mu_file);
        }
    }

    public function is_pro() {
        return $this->license && $this->license->is_valid();
    }
}

function safetypress() {
    return SafetyPress::instance();
}

// ============================================
// UPDATER INITIALISIEREN
// ============================================

if (class_exists('SafetyPress_Updater')) {
    new SafetyPress_Updater();
}

safetypress();
