<?php
if (!defined('ABSPATH')) exit;

class SafetyPress_Woo_Protection {

    public function __construct() {
        if (!class_exists('WooCommerce')) return;
        if (!get_option('safetypress_enable_woo_protection', 1)) return;
        
        add_action('woocommerce_checkout_order_processed', array($this, 'check_order'), 10, 3);
        add_action('woocommerce_admin_order_data_after_order_details', array($this, 'display_score'));
    }

    public function check_order($order_id, $posted_data, $order) {
        $score = $this->calculate_score($order);
        $factors = $this->get_factors($order);
        
        update_post_meta($order_id, '_safetypress_fraud_score', $score);
        update_post_meta($order_id, '_safetypress_risk_factors', $factors);
        
        $threshold = get_option('safetypress_fraud_score_threshold', 70);
        if ($score >= $threshold) {
            $order->update_status('on-hold', sprintf('SafetyPress: Hoher Betrugs-Score (%d)', $score));
        }
    }

    public function calculate_score($order) {
        $score = 0;
        $email = $order->get_billing_email();
        $billing = $order->get_billing_country();
        $shipping = $order->get_shipping_country();
        
        if ($this->is_disposable_email($email)) $score += 30;
        if ($billing && $shipping && $billing !== $shipping) $score += 20;
        if (in_array($billing, SafetyPress_Helper::get_high_risk_countries())) $score += 15;
        if ($order->get_total() > 500 && !$order->get_customer_id()) $score += 15;
        
        return min($score, 100);
    }

    public function get_factors($order) {
        $factors = array();
        $email = $order->get_billing_email();
        $billing = $order->get_billing_country();
        $shipping = $order->get_shipping_country();
        
        if ($this->is_disposable_email($email)) $factors[] = 'Einweg-E-Mail';
        if ($billing && $shipping && $billing !== $shipping) $factors[] = 'Länder unterschiedlich';
        if (in_array($billing, SafetyPress_Helper::get_high_risk_countries())) $factors[] = 'Hochrisiko-Land';
        if ($order->get_total() > 500 && !$order->get_customer_id()) $factors[] = 'Erste Bestellung, hoher Wert';
        
        return $factors;
    }

    private function is_disposable_email($email) {
        $domains = array('mailinator.com', 'tempmail.com', 'guerrillamail.com', '10minutemail.com');
        $domain = strtolower(substr(strrchr($email, '@'), 1));
        return in_array($domain, $domains);
    }

    public function display_score($order) {
        $score = get_post_meta($order->get_id(), '_safetypress_fraud_score', true);
        $factors = get_post_meta($order->get_id(), '_safetypress_risk_factors', true);
        if ($score === '') return;
        
        $color = $score >= 70 ? 'red' : ($score >= 50 ? 'orange' : 'green');
        ?>
        <div style="margin-top:20px;padding:15px;background:#f5f5f5;border-left:4px solid <?php echo $color; ?>;">
            <h4 style="margin:0 0 10px;">SafetyPress Betrugs-Score</h4>
            <p style="font-size:24px;font-weight:bold;color:<?php echo $color; ?>;margin:0 0 10px;"><?php echo esc_html($score); ?>/100</p>
            <?php if (!empty($factors)): ?>
                <p style="margin:0;"><strong>Risikofaktoren:</strong> <?php echo esc_html(implode(', ', $factors)); ?></p>
            <?php endif; ?>
        </div>
        <?php
    }
}
