<?php
/**
 * SafetyPress Auto-Updater
 * 
 * Prüft auf Updates vom eigenen Server.
 */

if (!defined('ABSPATH')) exit;

class SafetyPress_Updater {

    private $plugin_slug = 'safetypress';
    private $plugin_file = 'safetypress/safetypress.php';
    private $update_url = 'https://safetypress.de/api/license.php';
    private $cache_key = 'safetypress_update_data';
    private $cache_duration = 21600; // 6 Stunden

    public function __construct() {
        // Beim Lesen des Transients prüfen (wichtig!)
        add_filter('site_transient_update_plugins', [$this, 'check_for_update']);
        
        // Plugin Info für Details-Dialog
        add_filter('plugins_api', [$this, 'plugin_info'], 20, 3);
        
        // Cache leeren nach Update
        add_action('upgrader_process_complete', [$this, 'clear_cache'], 10, 2);
        
        // Manueller Cache-Clear via URL
        add_action('admin_init', [$this, 'maybe_clear_cache']);
    }

    /**
     * Prüft auf Updates - wird bei JEDEM Lesen des Transients aufgerufen
     */
    public function check_for_update($transient) {
        // Sicherstellen dass transient ein Objekt ist
        if (!is_object($transient)) {
            $transient = new stdClass();
        }
        
        // Wenn checked nicht gesetzt, zurückgeben
        if (empty($transient->checked)) {
            return $transient;
        }

        // Aktuelle Version holen
        $current_version = $transient->checked[$this->plugin_file] ?? '';
        
        if (empty($current_version)) {
            return $transient;
        }

        // Remote-Daten holen (mit Cache)
        $remote_data = $this->get_remote_data();
        
        if (!$remote_data || empty($remote_data->version)) {
            return $transient;
        }

        // Versions-Vergleich
        if (version_compare($current_version, $remote_data->version, '<')) {
            // Aus no_update entfernen falls vorhanden
            if (isset($transient->no_update[$this->plugin_file])) {
                unset($transient->no_update[$this->plugin_file]);
            }
            
            // Response-Array initialisieren falls nötig
            if (!isset($transient->response)) {
                $transient->response = [];
            }
            
            // Update-Info hinzufügen
            $transient->response[$this->plugin_file] = (object) [
                'id' => $this->plugin_file,
                'slug' => $this->plugin_slug,
                'plugin' => $this->plugin_file,
                'new_version' => $remote_data->version,
                'package' => $remote_data->download_url,
                'url' => 'https://safetypress.de',
                'tested' => $remote_data->tested ?? '',
                'requires_php' => $remote_data->requires_php ?? '7.4',
                'icons' => [
                    '1x' => 'https://safetypress.de/assets/icon-128x128.png',
                    '2x' => 'https://safetypress.de/assets/icon-256x256.png',
                    'default' => 'https://safetypress.de/assets/icon-256x256.png',
                ],
            ];
        } else {
            // Kein Update nötig - aus response entfernen falls vorhanden
            if (isset($transient->response[$this->plugin_file])) {
                unset($transient->response[$this->plugin_file]);
            }
        }

        return $transient;
    }

    /**
     * Plugin-Info für Details-Dialog
     */
    public function plugin_info($result, $action, $args) {
        if ($action !== 'plugin_information') {
            return $result;
        }
        
        if (!isset($args->slug) || $args->slug !== $this->plugin_slug) {
            return $result;
        }

        $remote_data = $this->get_plugin_info();

        if (!$remote_data) {
            return $result;
        }

        return (object) [
            'name' => $remote_data->name ?? 'SafetyPress',
            'slug' => $this->plugin_slug,
            'version' => $remote_data->version ?? '',
            'author' => '<a href="https://safetypress.de">SafetyPress</a>',
            'homepage' => $remote_data->homepage ?? 'https://safetypress.de',
            'requires' => $remote_data->requires ?? '5.8',
            'tested' => $remote_data->tested ?? '',
            'requires_php' => $remote_data->requires_php ?? '7.4',
            'download_link' => $remote_data->download_link ?? '',
            'trunk' => $remote_data->download_link ?? '',
            'last_updated' => $remote_data->last_updated ?? '',
            'sections' => [
                'description' => $remote_data->sections->description ?? 'WordPress & WooCommerce Security Plugin',
                'changelog' => $remote_data->sections->changelog ?? '',
            ],
            'icons' => [
                '1x' => 'https://safetypress.de/assets/icon-128x128.png',
                '2x' => 'https://safetypress.de/assets/icon-256x256.png',
                'default' => 'https://safetypress.de/assets/icon-256x256.png',
            ],
        ];
    }

    /**
     * Remote-Daten für Update-Check abrufen (mit Cache)
     */
    private function get_remote_data() {
        // Aus Cache holen
        $cached = get_transient($this->cache_key);
        
        if ($cached !== false && isset($cached->version)) {
            return $cached;
        }

        // Von API holen
        $response = wp_remote_get(
            $this->update_url . '?action=update_check',
            [
                'timeout' => 10,
                'sslverify' => true,
            ]
        );

        if (is_wp_error($response)) {
            return false;
        }
        
        if (wp_remote_retrieve_response_code($response) !== 200) {
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);

        if (empty($data) || !isset($data->version)) {
            return false;
        }

        // Im Cache speichern
        set_transient($this->cache_key, $data, $this->cache_duration);

        return $data;
    }

    /**
     * Plugin-Info für Details-Dialog abrufen
     */
    private function get_plugin_info() {
        $cache_key = $this->cache_key . '_info';
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }

        $response = wp_remote_get(
            $this->update_url . '?action=update_info',
            [
                'timeout' => 10,
                'sslverify' => true,
            ]
        );

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return false;
        }

        $data = json_decode(wp_remote_retrieve_body($response));

        if (empty($data)) {
            return false;
        }

        set_transient($cache_key, $data, $this->cache_duration);

        return $data;
    }

    /**
     * Cache leeren nach Update
     */
    public function clear_cache($upgrader, $options) {
        if (!isset($options['action']) || $options['action'] !== 'update') {
            return;
        }
        
        if (!isset($options['type']) || $options['type'] !== 'plugin') {
            return;
        }
        
        if (isset($options['plugins']) && in_array($this->plugin_file, $options['plugins'])) {
            delete_transient($this->cache_key);
            delete_transient($this->cache_key . '_info');
        }
    }

    /**
     * Cache manuell leeren via URL-Parameter
     */
    public function maybe_clear_cache() {
        if (!isset($_GET['safetypress_clear_update_cache'])) {
            return;
        }
        
        if (!current_user_can('manage_options')) {
            return;
        }

        // Alle relevanten Caches leeren
        delete_transient($this->cache_key);
        delete_transient($this->cache_key . '_info');
        delete_site_transient('update_plugins');
        wp_cache_delete('update_plugins', 'site-transients');
        
        // Weiterleiten ohne Parameter
        wp_safe_redirect(remove_query_arg('safetypress_clear_update_cache'));
        exit;
    }
}
