<?php
if (!defined('ABSPATH')) exit;

class SafetyPress_Two_Factor {

    private $secret_length = 16;
    private $code_length = 6;
    private $time_step = 30;
    private $algorithm = 'sha1';

    public function __construct() {
        add_action('init', array($this, 'maybe_init'));
    }

    public function maybe_init() {
        if (!$this->is_available()) return;
        
        // Profil-Seite
        add_action('show_user_profile', array($this, 'user_profile_setup'));
        add_action('edit_user_profile', array($this, 'user_profile_setup'));
        
        // Login-Hooks
        add_action('wp_login', array($this, 'check_2fa_on_login'), 10, 2);
        add_action('login_form_safetypress_2fa', array($this, 'render_2fa_login_form'));
        add_action('login_form_safetypress_2fa_verify', array($this, 'verify_2fa_login'));
        
        // AJAX-Handler
        add_action('wp_ajax_safetypress_2fa_setup', array($this, 'ajax_setup_2fa'));
        add_action('wp_ajax_safetypress_2fa_verify', array($this, 'ajax_verify_2fa'));
        add_action('wp_ajax_safetypress_2fa_disable', array($this, 'ajax_disable_2fa'));
        
        // Backup-Codes
        add_action('wp_ajax_safetypress_2fa_generate_backup', array($this, 'ajax_generate_backup_codes'));
    }

    public function is_available() {
        if (!get_option('safetypress_enable_2fa', 0)) return false;
        if (!function_exists('safetypress') || !safetypress()->is_pro()) return false;
        return true;
    }

    public function is_user_enabled($user_id) {
        return (bool) get_user_meta($user_id, '_safetypress_2fa_enabled', true);
    }

    public function get_user_secret($user_id) {
        return get_user_meta($user_id, '_safetypress_2fa_secret', true);
    }

    /**
     * Generiert einen zufälligen Base32-Secret
     */
    public function generate_secret() {
        $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567';
        $secret = '';
        
        for ($i = 0; $i < $this->secret_length; $i++) {
            $secret .= $chars[wp_rand(0, 31)];
        }
        
        return $secret;
    }

    /**
     * Base32-Dekodierung
     */
    private function base32_decode($secret) {
        $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567';
        $secret = strtoupper($secret);
        $n = 0;
        $j = 0;
        $binary = '';
        
        for ($i = 0; $i < strlen($secret); $i++) {
            $n = $n << 5;
            $n = $n + strpos($chars, $secret[$i]);
            $j = $j + 5;
            
            if ($j >= 8) {
                $j = $j - 8;
                $binary .= chr(($n >> $j) & 255);
            }
        }
        
        return $binary;
    }

    /**
     * Generiert einen TOTP-Code
     */
    public function generate_code($secret, $time = null) {
        if ($time === null) {
            $time = time();
        }
        
        $time_counter = floor($time / $this->time_step);
        $time_bytes = pack('N*', 0) . pack('N*', $time_counter);
        
        $secret_bytes = $this->base32_decode($secret);
        $hash = hash_hmac($this->algorithm, $time_bytes, $secret_bytes, true);
        
        $offset = ord(substr($hash, -1)) & 0x0F;
        $code = (
            ((ord($hash[$offset]) & 0x7F) << 24) |
            ((ord($hash[$offset + 1]) & 0xFF) << 16) |
            ((ord($hash[$offset + 2]) & 0xFF) << 8) |
            (ord($hash[$offset + 3]) & 0xFF)
        ) % pow(10, $this->code_length);
        
        return str_pad($code, $this->code_length, '0', STR_PAD_LEFT);
    }

    /**
     * Verifiziert einen TOTP-Code (mit Zeitfenster)
     */
    public function verify_code($secret, $code, $window = 1) {
        $code = preg_replace('/\s+/', '', $code);
        
        if (strlen($code) !== $this->code_length) {
            return false;
        }
        
        $time = time();
        
        for ($i = -$window; $i <= $window; $i++) {
            $check_time = $time + ($i * $this->time_step);
            $valid_code = $this->generate_code($secret, $check_time);
            
            if (hash_equals($valid_code, $code)) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Generiert die otpauth:// URL für QR-Code
     */
    public function get_qr_url($user, $secret) {
        $site_name = urlencode(get_bloginfo('name'));
        $account = urlencode($user->user_email);
        
        return sprintf(
            'otpauth://totp/%s:%s?secret=%s&issuer=%s&algorithm=SHA1&digits=6&period=30',
            $site_name,
            $account,
            $secret,
            $site_name
        );
    }

    /**
     * Generiert Backup-Codes
     */
    public function generate_backup_codes($user_id, $count = 10) {
        $codes = array();
        
        for ($i = 0; $i < $count; $i++) {
            $codes[] = strtoupper(wp_generate_password(4, false) . '-' . wp_generate_password(4, false));
        }
        
        // Hash die Codes vor dem Speichern
        $hashed_codes = array_map(function($code) {
            return wp_hash_password($code);
        }, $codes);
        
        update_user_meta($user_id, '_safetypress_2fa_backup_codes', $hashed_codes);
        
        return $codes; // Klartext für einmalige Anzeige
    }

    /**
     * Prüft einen Backup-Code
     */
    public function verify_backup_code($user_id, $code) {
        $code = strtoupper(preg_replace('/\s+/', '', $code));
        $stored_codes = get_user_meta($user_id, '_safetypress_2fa_backup_codes', true);
        
        if (!is_array($stored_codes) || empty($stored_codes)) {
            return false;
        }
        
        foreach ($stored_codes as $index => $hashed_code) {
            if (wp_check_password($code, $hashed_code)) {
                // Code wurde verwendet - entfernen
                unset($stored_codes[$index]);
                update_user_meta($user_id, '_safetypress_2fa_backup_codes', array_values($stored_codes));
                return true;
            }
        }
        
        return false;
    }

    /**
     * Profil-Seite: 2FA Setup
     */
    public function user_profile_setup($user) {
        if (!current_user_can('edit_user', $user->ID)) return;
        
        $is_enabled = $this->is_user_enabled($user->ID);
        $pending_secret = get_user_meta($user->ID, '_safetypress_2fa_pending_secret', true);
        $backup_count = 0;
        
        if ($is_enabled) {
            $backup_codes = get_user_meta($user->ID, '_safetypress_2fa_backup_codes', true);
            $backup_count = is_array($backup_codes) ? count($backup_codes) : 0;
        }
        
        ?>
        <h2 style="margin-top: 40px;">🔐 Zwei-Faktor-Authentifizierung</h2>
        <p class="description">Schützen Sie Ihr Konto mit einem zusätzlichen Sicherheitsfaktor.</p>
        
        <table class="form-table" id="safetypress-2fa-table">
            <tr>
                <th>Status</th>
                <td>
                    <div id="safetypress-2fa-status">
                        <?php if ($is_enabled): ?>
                            <span style="color: #00a32a; font-weight: 600;">✓ 2FA ist aktiviert</span>
                            <p class="description" style="margin-top: 8px;">
                                Backup-Codes verfügbar: <strong><?php echo $backup_count; ?></strong>
                            </p>
                        <?php else: ?>
                            <span style="color: #72777c;">✗ Nicht aktiviert</span>
                        <?php endif; ?>
                    </div>
                </td>
            </tr>
            
            <?php if (!$is_enabled): ?>
            <tr id="safetypress-2fa-setup-row">
                <th>Einrichten</th>
                <td>
                    <button type="button" class="button button-primary" id="safetypress-2fa-start-setup">
                        2FA aktivieren
                    </button>
                    
                    <div id="safetypress-2fa-setup-container" style="display: none; margin-top: 20px; max-width: 500px;">
                        <div style="background: #f6f7f7; padding: 20px; border-radius: 4px; border: 1px solid #c3c4c7;">
                            <h4 style="margin-top: 0;">Schritt 1: App scannen</h4>
                            <p>Scannen Sie diesen QR-Code mit Ihrer Authenticator-App (z.B. Google Authenticator, Authy):</p>
                            
                            <div id="safetypress-2fa-qr" style="text-align: center; margin: 20px 0; background: #fff; padding: 20px; display: inline-block;">
                                <!-- QR Code wird via JS eingefügt -->
                            </div>
                            
                            <p style="font-size: 12px; color: #646970;">
                                Oder geben Sie diesen Code manuell ein:<br>
                                <code id="safetypress-2fa-secret" style="font-size: 14px; letter-spacing: 2px;"></code>
                            </p>
                            
                            <h4>Schritt 2: Code bestätigen</h4>
                            <p>Geben Sie den 6-stelligen Code aus Ihrer App ein:</p>
                            
                            <input type="text" id="safetypress-2fa-verify-code" 
                                   placeholder="000000" 
                                   maxlength="6" 
                                   pattern="[0-9]*"
                                   inputmode="numeric"
                                   style="width: 150px; text-align: center; font-size: 24px; letter-spacing: 8px; padding: 10px;">
                            
                            <div style="margin-top: 15px;">
                                <button type="button" class="button button-primary" id="safetypress-2fa-confirm">
                                    Bestätigen & Aktivieren
                                </button>
                                <button type="button" class="button" id="safetypress-2fa-cancel" style="margin-left: 10px;">
                                    Abbrechen
                                </button>
                            </div>
                            
                            <div id="safetypress-2fa-message" style="margin-top: 15px;"></div>
                        </div>
                    </div>
                </td>
            </tr>
            <?php else: ?>
            <tr>
                <th>Backup-Codes</th>
                <td>
                    <button type="button" class="button" id="safetypress-2fa-new-backup">
                        Neue Backup-Codes generieren
                    </button>
                    <p class="description">Falls Sie keinen Zugriff auf Ihre App haben, können Sie einen Backup-Code verwenden.</p>
                    
                    <div id="safetypress-2fa-backup-container" style="display: none; margin-top: 15px;">
                        <div style="background: #fcf0f1; padding: 15px; border-radius: 4px; border: 1px solid #d63638;">
                            <strong style="color: #d63638;">⚠️ Wichtig: Speichern Sie diese Codes sicher!</strong>
                            <p>Diese Codes werden nur einmal angezeigt. Jeder Code kann nur einmal verwendet werden.</p>
                            <div id="safetypress-2fa-backup-codes" style="font-family: monospace; background: #fff; padding: 15px; margin-top: 10px; border-radius: 4px;"></div>
                        </div>
                    </div>
                </td>
            </tr>
            <tr>
                <th>Deaktivieren</th>
                <td>
                    <button type="button" class="button" id="safetypress-2fa-disable" style="color: #d63638; border-color: #d63638;">
                        2FA deaktivieren
                    </button>
                    <p class="description">Sie müssen Ihren aktuellen Code eingeben, um 2FA zu deaktivieren.</p>
                </td>
            </tr>
            <?php endif; ?>
        </table>
        
        <script>
        jQuery(document).ready(function($) {
            var pendingSecret = '';
            var userId = <?php echo $user->ID; ?>;
            
            // Setup starten
            $('#safetypress-2fa-start-setup').on('click', function() {
                var $btn = $(this);
                $btn.prop('disabled', true).text('Wird vorbereitet...');
                
                $.post(ajaxurl, {
                    action: 'safetypress_2fa_setup',
                    user_id: userId,
                    nonce: '<?php echo wp_create_nonce('safetypress_2fa_setup'); ?>'
                }, function(response) {
                    if (response.success) {
                        pendingSecret = response.data.secret;
                        $('#safetypress-2fa-secret').text(response.data.secret);
                        $('#safetypress-2fa-qr').html('<img src="' + response.data.qr_url + '" alt="QR Code" style="max-width: 200px;">');
                        $('#safetypress-2fa-setup-container').slideDown();
                        $btn.hide();
                    } else {
                        alert(response.data.message || 'Fehler beim Setup');
                        $btn.prop('disabled', false).text('2FA aktivieren');
                    }
                });
            });
            
            // Setup abbrechen
            $('#safetypress-2fa-cancel').on('click', function() {
                $('#safetypress-2fa-setup-container').slideUp();
                $('#safetypress-2fa-start-setup').show().prop('disabled', false).text('2FA aktivieren');
                $('#safetypress-2fa-verify-code').val('');
                $('#safetypress-2fa-message').html('');
            });
            
            // Code bestätigen
            $('#safetypress-2fa-confirm').on('click', function() {
                var code = $('#safetypress-2fa-verify-code').val();
                var $btn = $(this);
                var $msg = $('#safetypress-2fa-message');
                
                if (!code || code.length !== 6) {
                    $msg.html('<p style="color: #d63638;">Bitte geben Sie einen 6-stelligen Code ein.</p>');
                    return;
                }
                
                $btn.prop('disabled', true).text('Wird überprüft...');
                
                $.post(ajaxurl, {
                    action: 'safetypress_2fa_verify',
                    user_id: userId,
                    code: code,
                    secret: pendingSecret,
                    nonce: '<?php echo wp_create_nonce('safetypress_2fa_verify'); ?>'
                }, function(response) {
                    if (response.success) {
                        $msg.html('<p style="color: #00a32a;">✓ 2FA erfolgreich aktiviert!</p>');
                        
                        // Backup-Codes anzeigen
                        if (response.data.backup_codes) {
                            var codesHtml = '<strong>Ihre Backup-Codes:</strong><br><br>';
                            response.data.backup_codes.forEach(function(code) {
                                codesHtml += '<code style="display: inline-block; margin: 3px 5px; padding: 5px 10px; background: #f0f0f1;">' + code + '</code>';
                            });
                            $msg.append('<div style="margin-top: 15px; background: #fcf0f1; padding: 15px; border-radius: 4px;">' +
                                '<strong style="color: #d63638;">⚠️ Speichern Sie diese Codes!</strong><br><br>' + codesHtml + '</div>');
                        }
                        
                        setTimeout(function() { location.reload(); }, 5000);
                    } else {
                        $msg.html('<p style="color: #d63638;">' + (response.data.message || 'Ungültiger Code') + '</p>');
                        $btn.prop('disabled', false).text('Bestätigen & Aktivieren');
                    }
                });
            });
            
            // 2FA deaktivieren
            $('#safetypress-2fa-disable').on('click', function() {
                var code = prompt('Geben Sie Ihren aktuellen 2FA-Code ein:');
                if (!code) return;
                
                var $btn = $(this);
                $btn.prop('disabled', true);
                
                $.post(ajaxurl, {
                    action: 'safetypress_2fa_disable',
                    user_id: userId,
                    code: code,
                    nonce: '<?php echo wp_create_nonce('safetypress_2fa_disable'); ?>'
                }, function(response) {
                    if (response.success) {
                        alert('2FA wurde deaktiviert.');
                        location.reload();
                    } else {
                        alert(response.data.message || 'Fehler beim Deaktivieren');
                        $btn.prop('disabled', false);
                    }
                });
            });
            
            // Neue Backup-Codes
            $('#safetypress-2fa-new-backup').on('click', function() {
                if (!confirm('Alle bisherigen Backup-Codes werden ungültig. Fortfahren?')) return;
                
                var $btn = $(this);
                $btn.prop('disabled', true);
                
                $.post(ajaxurl, {
                    action: 'safetypress_2fa_generate_backup',
                    user_id: userId,
                    nonce: '<?php echo wp_create_nonce('safetypress_2fa_backup'); ?>'
                }, function(response) {
                    if (response.success) {
                        var codesHtml = '';
                        response.data.codes.forEach(function(code) {
                            codesHtml += '<code style="display: inline-block; margin: 3px 5px; padding: 5px 10px; background: #f0f0f1;">' + code + '</code>';
                        });
                        $('#safetypress-2fa-backup-codes').html(codesHtml);
                        $('#safetypress-2fa-backup-container').slideDown();
                    } else {
                        alert(response.data.message || 'Fehler');
                    }
                    $btn.prop('disabled', false);
                });
            });
            
            // Auto-Focus und Enter-Handler
            $('#safetypress-2fa-verify-code').on('keypress', function(e) {
                if (e.which === 13) {
                    e.preventDefault();
                    $('#safetypress-2fa-confirm').click();
                }
            });
        });
        </script>
        <?php
    }

    /**
     * AJAX: 2FA Setup starten
     */
    public function ajax_setup_2fa() {
        check_ajax_referer('safetypress_2fa_setup', 'nonce');
        
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        
        if (!$user_id || !current_user_can('edit_user', $user_id)) {
            wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        }
        
        $user = get_user_by('ID', $user_id);
        if (!$user) {
            wp_send_json_error(array('message' => 'Benutzer nicht gefunden.'));
        }
        
        $secret = $this->generate_secret();
        $qr_url = $this->get_qr_url($user, $secret);
        
        // QR-Code via Google Charts API
        $qr_image = 'https://chart.googleapis.com/chart?chs=200x200&chld=M|0&cht=qr&chl=' . urlencode($qr_url);
        
        // Pending Secret speichern
        update_user_meta($user_id, '_safetypress_2fa_pending_secret', $secret);
        
        wp_send_json_success(array(
            'secret' => $secret,
            'qr_url' => $qr_image,
        ));
    }

    /**
     * AJAX: 2FA Code verifizieren und aktivieren
     */
    public function ajax_verify_2fa() {
        check_ajax_referer('safetypress_2fa_verify', 'nonce');
        
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        $code = isset($_POST['code']) ? sanitize_text_field($_POST['code']) : '';
        $secret = isset($_POST['secret']) ? sanitize_text_field($_POST['secret']) : '';
        
        if (!$user_id || !current_user_can('edit_user', $user_id)) {
            wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        }
        
        // Pending Secret prüfen
        $pending_secret = get_user_meta($user_id, '_safetypress_2fa_pending_secret', true);
        if ($secret !== $pending_secret) {
            wp_send_json_error(array('message' => 'Ungültiger Secret-Key.'));
        }
        
        // Code verifizieren
        if (!$this->verify_code($secret, $code)) {
            wp_send_json_error(array('message' => 'Ungültiger Code. Bitte versuchen Sie es erneut.'));
        }
        
        // 2FA aktivieren
        update_user_meta($user_id, '_safetypress_2fa_secret', $secret);
        update_user_meta($user_id, '_safetypress_2fa_enabled', 1);
        delete_user_meta($user_id, '_safetypress_2fa_pending_secret');
        
        // Backup-Codes generieren
        $backup_codes = $this->generate_backup_codes($user_id);
        
        // Log
        if (function_exists('safetypress') && safetypress()->logger) {
            $user = get_user_by('ID', $user_id);
            safetypress()->logger->log('2fa_enabled', sprintf('2FA aktiviert für %s', $user->user_login), array(
                'user_id' => $user_id,
            ));
        }
        
        wp_send_json_success(array(
            'message' => '2FA erfolgreich aktiviert!',
            'backup_codes' => $backup_codes,
        ));
    }

    /**
     * AJAX: 2FA deaktivieren
     */
    public function ajax_disable_2fa() {
        check_ajax_referer('safetypress_2fa_disable', 'nonce');
        
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        $code = isset($_POST['code']) ? sanitize_text_field($_POST['code']) : '';
        
        if (!$user_id || !current_user_can('edit_user', $user_id)) {
            wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        }
        
        $secret = $this->get_user_secret($user_id);
        
        // Code oder Backup-Code verifizieren
        if (!$this->verify_code($secret, $code) && !$this->verify_backup_code($user_id, $code)) {
            wp_send_json_error(array('message' => 'Ungültiger Code.'));
        }
        
        // 2FA deaktivieren
        delete_user_meta($user_id, '_safetypress_2fa_enabled');
        delete_user_meta($user_id, '_safetypress_2fa_secret');
        delete_user_meta($user_id, '_safetypress_2fa_backup_codes');
        delete_user_meta($user_id, '_safetypress_2fa_pending_secret');
        
        // Log
        if (function_exists('safetypress') && safetypress()->logger) {
            $user = get_user_by('ID', $user_id);
            safetypress()->logger->log('2fa_disabled', sprintf('2FA deaktiviert für %s', $user->user_login), array(
                'user_id' => $user_id,
            ));
        }
        
        wp_send_json_success(array('message' => '2FA wurde deaktiviert.'));
    }

    /**
     * AJAX: Neue Backup-Codes generieren
     */
    public function ajax_generate_backup_codes() {
        check_ajax_referer('safetypress_2fa_backup', 'nonce');
        
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        
        if (!$user_id || !current_user_can('edit_user', $user_id)) {
            wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        }
        
        if (!$this->is_user_enabled($user_id)) {
            wp_send_json_error(array('message' => '2FA ist nicht aktiviert.'));
        }
        
        $codes = $this->generate_backup_codes($user_id);
        
        wp_send_json_success(array('codes' => $codes));
    }

    /**
     * Login-Check: 2FA erforderlich?
     */
    public function check_2fa_on_login($user_login, $user) {
        if (!$this->is_user_enabled($user->ID)) {
            return;
        }
        
        // Session-Token für 2FA-Verifizierung speichern
        $token = wp_generate_password(32, false);
        set_transient('safetypress_2fa_login_' . $token, array(
            'user_id' => $user->ID,
            'time' => time(),
        ), 300); // 5 Minuten gültig
        
        // Benutzer ausloggen und zur 2FA-Seite weiterleiten
        wp_clear_auth_cookie();
        
        $redirect_url = add_query_arg(array(
            'action' => 'safetypress_2fa',
            'token' => $token,
            'redirect_to' => isset($_REQUEST['redirect_to']) ? urlencode($_REQUEST['redirect_to']) : '',
        ), wp_login_url());
        
        wp_safe_redirect($redirect_url);
        exit;
    }

    /**
     * 2FA Login-Formular anzeigen
     */
    public function render_2fa_login_form() {
        $token = isset($_GET['token']) ? sanitize_text_field($_GET['token']) : '';
        $redirect_to = isset($_GET['redirect_to']) ? esc_url(urldecode($_GET['redirect_to'])) : admin_url();
        $error = isset($_GET['error']) ? sanitize_text_field($_GET['error']) : '';
        
        $login_data = get_transient('safetypress_2fa_login_' . $token);
        
        if (!$login_data) {
            wp_safe_redirect(wp_login_url());
            exit;
        }
        
        $user = get_user_by('ID', $login_data['user_id']);
        
        login_header(__('Zwei-Faktor-Authentifizierung', 'safetypress'), '', array());
        ?>
        <form name="2fa_form" id="2fa_form" action="<?php echo esc_url(wp_login_url() . '?action=safetypress_2fa_verify'); ?>" method="post">
            <input type="hidden" name="token" value="<?php echo esc_attr($token); ?>">
            <input type="hidden" name="redirect_to" value="<?php echo esc_attr($redirect_to); ?>">
            <?php wp_nonce_field('safetypress_2fa_login', 'safetypress_2fa_nonce'); ?>
            
            <div style="text-align: center; margin-bottom: 20px;">
                <span style="font-size: 48px;">🔐</span>
            </div>
            
            <p style="text-align: center; margin-bottom: 20px;">
                Hallo <strong><?php echo esc_html($user->display_name); ?></strong>,<br>
                bitte geben Sie Ihren 2FA-Code ein.
            </p>
            
            <?php if ($error === 'invalid'): ?>
            <div id="login_error" style="margin-bottom: 16px;">
                <strong>Fehler:</strong> Ungültiger Code. Bitte versuchen Sie es erneut.
            </div>
            <?php endif; ?>
            
            <p>
                <label for="2fa_code">Authenticator-Code</label>
                <input type="text" name="2fa_code" id="2fa_code" class="input" 
                       placeholder="000000" 
                       maxlength="10"
                       pattern="[0-9A-Za-z\-]*"
                       autocomplete="one-time-code"
                       autofocus
                       style="text-align: center; font-size: 20px; letter-spacing: 5px;">
            </p>
            
            <p class="submit">
                <input type="submit" name="wp-submit" id="wp-submit" class="button button-primary button-large" value="Verifizieren">
            </p>
            
            <p style="text-align: center; font-size: 13px; color: #72777c;">
                Sie können auch einen Backup-Code verwenden.
            </p>
        </form>
        
        <p id="backtoblog">
            <a href="<?php echo esc_url(wp_login_url()); ?>">&larr; Zurück zum Login</a>
        </p>
        
        <script>
        document.getElementById('2fa_code').focus();
        </script>
        <?php
        login_footer();
        exit;
    }

    /**
     * 2FA Login verifizieren
     */
    public function verify_2fa_login() {
        if (!isset($_POST['safetypress_2fa_nonce']) || !wp_verify_nonce($_POST['safetypress_2fa_nonce'], 'safetypress_2fa_login')) {
            wp_safe_redirect(wp_login_url());
            exit;
        }
        
        $token = isset($_POST['token']) ? sanitize_text_field($_POST['token']) : '';
        $code = isset($_POST['2fa_code']) ? sanitize_text_field($_POST['2fa_code']) : '';
        $redirect_to = isset($_POST['redirect_to']) ? esc_url($_POST['redirect_to']) : admin_url();
        
        $login_data = get_transient('safetypress_2fa_login_' . $token);
        
        if (!$login_data) {
            wp_safe_redirect(wp_login_url());
            exit;
        }
        
        $user_id = $login_data['user_id'];
        $secret = $this->get_user_secret($user_id);
        
        // TOTP-Code oder Backup-Code prüfen
        $valid = $this->verify_code($secret, $code) || $this->verify_backup_code($user_id, $code);
        
        if (!$valid) {
            // Log fehlgeschlagenen Versuch
            if (function_exists('safetypress') && safetypress()->logger) {
                $user = get_user_by('ID', $user_id);
                safetypress()->logger->log('2fa_failed', sprintf('2FA-Verifizierung fehlgeschlagen für %s', $user->user_login), array(
                    'user_id' => $user_id,
                ));
            }
            
            wp_safe_redirect(add_query_arg(array(
                'action' => 'safetypress_2fa',
                'token' => $token,
                'redirect_to' => urlencode($redirect_to),
                'error' => 'invalid',
            ), wp_login_url()));
            exit;
        }
        
        // Erfolgreich - Token löschen und einloggen
        delete_transient('safetypress_2fa_login_' . $token);
        
        $user = get_user_by('ID', $user_id);
        wp_set_auth_cookie($user_id, true);
        
        // Log
        if (function_exists('safetypress') && safetypress()->logger) {
            safetypress()->logger->log('2fa_success', sprintf('2FA-Login erfolgreich für %s', $user->user_login), array(
                'user_id' => $user_id,
            ));
        }
        
        wp_safe_redirect($redirect_to);
        exit;
    }
}
