<?php
/**
 * SafetyPress Stripe Integration
 * 
 * Kommuniziert mit der SafetyPress License API.
 * KEINE Stripe-Keys im Plugin - alles läuft über die API!
 */

if (!defined('ABSPATH')) exit;

class SafetyPress_Stripe {

    private $api_base = 'https://safetypress.de/api';
    private $cache_key = 'safetypress_license_cache';
    private $cache_duration = 3600; // 1 Stunde

    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialisiere Hooks
     */
    private function init_hooks() {
        // AJAX Handlers
        add_action('wp_ajax_safetypress_create_checkout', array($this, 'ajax_create_checkout'));
        add_action('wp_ajax_safetypress_get_portal_url', array($this, 'ajax_get_portal_url'));
        add_action('wp_ajax_safetypress_check_subscription', array($this, 'ajax_check_subscription'));
        
        // Lizenz-Check bei Admin-Load
        add_action('admin_init', array($this, 'maybe_check_license'));
        
        // Checkout-Success Handler
        add_action('admin_init', array($this, 'handle_checkout_return'));
    }

    /**
     * API Request
     */
    private function api_request($endpoint, $data = array()) {
        $url = $this->api_base . '/' . ltrim($endpoint, '/');
        
        // Domain immer mitsenden
        $data['domain'] = home_url();
        
        $response = wp_remote_post($url, array(
            'timeout' => 30,
            'body' => $data,
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
            ),
        ));
        
        if (is_wp_error($response)) {
            return array('error' => true, 'message' => $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $code = wp_remote_retrieve_response_code($response);
        
        if ($code >= 400) {
            return array(
                'error' => true, 
                'message' => isset($body['error']) ? $body['error'] : 'API Error',
                'code' => $code
            );
        }
        
        return $body;
    }

    /**
     * Lizenz bei API prüfen
     */
    public function check_license($force = false) {
        // Cache prüfen
        if (!$force) {
            $cached = get_transient($this->cache_key);
            if ($cached !== false) {
                return $cached;
            }
        }
        
        $result = $this->api_request('license.php', array('action' => 'check'));
        
        if (!isset($result['error'])) {
            // In Datenbank speichern für Offline-Zugriff
            $this->save_license_locally($result);
            
            // Cache setzen
            set_transient($this->cache_key, $result, $this->cache_duration);
        }
        
        return $result;
    }

    /**
     * Lizenz lokal speichern
     */
    private function save_license_locally($data) {
        if (isset($data['valid']) && $data['valid']) {
            update_option('safetypress_license_status', $data['status']);
            update_option('safetypress_license_type', $data['type'] ?? 'subscription');
            update_option('safetypress_license_tier', $data['tier'] ?? 1);
            update_option('safetypress_stripe_subscription_status', $data['status']);
            
            if (!empty($data['expires'])) {
                update_option('safetypress_license_expires', $data['expires']);
            }
            if (!empty($data['warning'])) {
                update_option('safetypress_license_warning', $data['warning']);
            } else {
                delete_option('safetypress_license_warning');
            }
            if (!empty($data['activated'])) {
                update_option('safetypress_license_activated', $data['activated']);
            }
            if (!empty($data['license_key'])) {
                update_option('safetypress_license_key', $data['license_key']);
            }
        } else {
            // Keine gültige Lizenz
            update_option('safetypress_license_status', 'free');
            delete_option('safetypress_stripe_subscription_status');
        }
    }

    /**
     * Periodisch Lizenz prüfen
     */
    public function maybe_check_license() {
        // Nur auf SafetyPress-Seiten
        if (!isset($_GET['page']) || strpos($_GET['page'], 'safetypress') !== 0) {
            return;
        }
        
        // Nicht bei jedem Request, nur wenn Cache abgelaufen
        $cached = get_transient($this->cache_key);
        if ($cached === false) {
            $this->check_license(true);
        }
    }

    /**
     * Handle Checkout Return
     */
    public function handle_checkout_return() {
        if (!isset($_GET['page']) || $_GET['page'] !== 'safetypress-license') {
            return;
        }
        
        if (isset($_GET['checkout']) && $_GET['checkout'] === 'success') {
            // Cache leeren und neu prüfen
            delete_transient($this->cache_key);
            
            // Kurz warten damit Webhook verarbeitet wird
            sleep(2);
            
            // Lizenz neu prüfen
            $license = $this->check_license(true);
            
            if (isset($license['valid']) && $license['valid']) {
                // Redirect ohne Query-Parameter
                wp_safe_redirect(admin_url('admin.php?page=safetypress-license&activated=1'));
                exit;
            }
        }
    }

    /**
     * Lizenz mit Key aktivieren
     */
    public function activate_license($key) {
        $result = $this->api_request('license.php', array(
            'action' => 'activate',
            'license_key' => $key
        ));
        
        if (isset($result['success']) && $result['success']) {
            // Cache leeren
            delete_transient($this->cache_key);
            
            // Neu prüfen
            $this->check_license(true);
        }
        
        return $result;
    }

    /**
     * Lizenz deaktivieren
     */
    public function deactivate_license() {
        $result = $this->api_request('license.php', array('action' => 'deactivate'));
        
        // Lokale Daten löschen
        delete_option('safetypress_license_key');
        update_option('safetypress_license_status', 'free');
        delete_option('safetypress_license_type');
        delete_option('safetypress_license_tier');
        delete_option('safetypress_stripe_subscription_status');
        delete_option('safetypress_license_warning');
        delete_option('safetypress_license_expires');
        delete_transient($this->cache_key);
        
        return $result;
    }

    /**
     * Checkout Session erstellen
     */
    public function create_checkout_session($email = '') {
        $current_user = wp_get_current_user();
        
        $data = array(
            'email' => !empty($email) ? $email : $current_user->user_email,
            'success_url' => admin_url('admin.php?page=safetypress-license&checkout=success&session_id={CHECKOUT_SESSION_ID}'),
            'cancel_url' => admin_url('admin.php?page=safetypress-license&checkout=cancelled'),
        );
        
        return $this->api_request('checkout.php', $data);
    }

    /**
     * Customer Portal URL holen
     */
    public function get_portal_url() {
        return $this->api_request('portal.php', array(
            'return_url' => admin_url('admin.php?page=safetypress-license')
        ));
    }

    /**
     * AJAX: Checkout Session erstellen
     */
    public function ajax_create_checkout() {
        check_ajax_referer('safetypress_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        }
        
        $email = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';
        $result = $this->create_checkout_session($email);
        
        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['message']));
        }
        
        wp_send_json_success(array(
            'checkout_url' => $result['checkout_url'],
            'session_id' => $result['session_id'],
        ));
    }

    /**
     * AJAX: Portal URL holen
     */
    public function ajax_get_portal_url() {
        check_ajax_referer('safetypress_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        }
        
        $result = $this->get_portal_url();
        
        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['message']));
        }
        
        wp_send_json_success(array('portal_url' => $result['portal_url']));
    }

    /**
     * AJAX: Subscription Status prüfen
     */
    public function ajax_check_subscription() {
        check_ajax_referer('safetypress_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        }
        
        $result = $this->check_license(true);
        
        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['message']));
        }
        
        wp_send_json_success($result);
    }

    /**
     * Hole Subscription-Info für Templates
     */
    public function get_subscription_info() {
        $status = get_option('safetypress_stripe_subscription_status', '');
        
        return array(
            'has_subscription' => in_array($status, array('active', 'trialing')),
            'status' => $status,
            'warning' => get_option('safetypress_license_warning', ''),
            'expires' => get_option('safetypress_license_expires', ''),
            'is_active' => in_array($status, array('active', 'trialing')),
        );
    }

    /**
     * Ist eine aktive Subscription vorhanden?
     */
    public function has_active_subscription() {
        $info = $this->get_subscription_info();
        return $info['is_active'];
    }
}
