<?php
/**
 * SafetyPress Sicherheits-Score
 * 
 * Berechnet einen Sicherheits-Score und gibt Empfehlungen.
 */

if (!defined('ABSPATH')) exit;

class SafetyPress_Security_Score {

    private $checks = [];
    private $score = 0;
    private $max_score = 0;

    public function __construct() {
        add_action('wp_ajax_safetypress_get_security_score', [$this, 'ajax_get_score']);
    }

    /**
     * Sicherheits-Score berechnen
     */
    public function calculate_score() {
        $this->checks = [];
        $this->score = 0;
        $this->max_score = 0;
        
        // Alle Checks durchführen
        $this->check_ssl();
        $this->check_wp_version();
        $this->check_php_version();
        $this->check_debug_mode();
        $this->check_file_editor();
        $this->check_xmlrpc();
        $this->check_login_protection();
        $this->check_firewall();
        $this->check_two_factor();
        $this->check_security_headers();
        $this->check_database_prefix();
        $this->check_admin_username();
        $this->check_file_permissions();
        $this->check_plugins_updates();
        $this->check_themes_updates();
        $this->check_inactive_plugins();
        $this->check_login_url();
        $this->check_scheduled_scans();
        $this->check_integrity();
        $this->check_backup();
        
        // Prozentwert berechnen
        $percentage = $this->max_score > 0 ? round(($this->score / $this->max_score) * 100) : 0;
        
        return [
            'score' => $this->score,
            'max_score' => $this->max_score,
            'percentage' => $percentage,
            'grade' => $this->get_grade($percentage),
            'checks' => $this->checks,
            'recommendations' => $this->get_recommendations(),
        ];
    }

    /**
     * Check hinzufügen
     */
    private function add_check($id, $name, $passed, $weight, $description, $recommendation = '') {
        $this->max_score += $weight;
        
        if ($passed) {
            $this->score += $weight;
        }
        
        $this->checks[$id] = [
            'name' => $name,
            'passed' => $passed,
            'weight' => $weight,
            'description' => $description,
            'recommendation' => $recommendation,
        ];
    }

    // ========================================
    // EINZELNE CHECKS
    // ========================================

    private function check_ssl() {
        $passed = is_ssl();
        $this->add_check(
            'ssl',
            'SSL/HTTPS aktiviert',
            $passed,
            15,
            'Verschlüsselte Verbindung zur Website',
            $passed ? '' : 'Aktivieren Sie ein SSL-Zertifikat bei Ihrem Hoster.'
        );
    }

    private function check_wp_version() {
        global $wp_version;
        $response = wp_remote_get('https://api.wordpress.org/core/version-check/1.7/');
        $latest = '';
        
        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            $latest = $data['offers'][0]['version'] ?? '';
        }
        
        $passed = empty($latest) || version_compare($wp_version, $latest, '>=');
        
        $this->add_check(
            'wp_version',
            'WordPress aktuell',
            $passed,
            10,
            'WordPress Version: ' . $wp_version,
            $passed ? '' : "Aktualisieren Sie auf WordPress {$latest}."
        );
    }

    private function check_php_version() {
        $version = PHP_VERSION;
        $passed = version_compare($version, '8.0', '>=');
        
        $this->add_check(
            'php_version',
            'PHP Version aktuell',
            $passed,
            8,
            'PHP Version: ' . $version,
            $passed ? '' : 'Aktualisieren Sie PHP auf mindestens Version 8.0.'
        );
    }

    private function check_debug_mode() {
        $passed = !defined('WP_DEBUG') || !WP_DEBUG;
        
        $this->add_check(
            'debug_mode',
            'Debug-Modus deaktiviert',
            $passed,
            5,
            'WP_DEBUG sollte auf Produktionsseiten deaktiviert sein',
            $passed ? '' : 'Setzen Sie WP_DEBUG auf false in wp-config.php.'
        );
    }

    private function check_file_editor() {
        $passed = defined('DISALLOW_FILE_EDIT') && DISALLOW_FILE_EDIT;
        
        $this->add_check(
            'file_editor',
            'Datei-Editor deaktiviert',
            $passed,
            8,
            'Verhindert Code-Bearbeitung im Admin',
            $passed ? '' : 'Aktivieren Sie diese Option in SafetyPress Hardening.'
        );
    }

    private function check_xmlrpc() {
        $passed = (bool) get_option('safetypress_disable_xmlrpc', 1);
        
        $this->add_check(
            'xmlrpc',
            'XML-RPC deaktiviert',
            $passed,
            7,
            'Verhindert Remote-Angriffe via XML-RPC',
            $passed ? '' : 'Aktivieren Sie diese Option in SafetyPress Hardening.'
        );
    }

    private function check_login_protection() {
        $passed = (bool) get_option('safetypress_enable_login_protection', 1);
        
        $this->add_check(
            'login_protection',
            'Login-Schutz aktiviert',
            $passed,
            10,
            'Brute-Force Schutz für die Login-Seite',
            $passed ? '' : 'Aktivieren Sie den Login-Schutz in SafetyPress.'
        );
    }

    private function check_firewall() {
        $passed = (bool) get_option('safetypress_enable_firewall', 1);
        
        $this->add_check(
            'firewall',
            'Firewall aktiviert',
            $passed,
            12,
            'Schützt vor böswilligen Anfragen',
            $passed ? '' : 'Aktivieren Sie die Firewall in SafetyPress.'
        );
    }

    private function check_two_factor() {
        $passed = (bool) get_option('safetypress_enable_2fa', 0);
        
        $this->add_check(
            'two_factor',
            '2-Faktor-Authentifizierung',
            $passed,
            10,
            'Extra Sicherheitsebene beim Login',
            $passed ? '' : 'Aktivieren Sie 2FA für zusätzliche Sicherheit.'
        );
    }

    private function check_security_headers() {
        // Immer aktiv bei SafetyPress
        $this->add_check(
            'security_headers',
            'Security Headers aktiv',
            true,
            8,
            'HTTP-Sicherheits-Header werden gesendet',
            ''
        );
    }

    private function check_database_prefix() {
        global $wpdb;
        $passed = $wpdb->prefix !== 'wp_';
        
        $this->add_check(
            'db_prefix',
            'Datenbank-Präfix geändert',
            $passed,
            3,
            'Standard-Präfix: ' . $wpdb->prefix,
            $passed ? '' : 'Ein benutzerdefiniertes Präfix erhöht die Sicherheit leicht.'
        );
    }

    private function check_admin_username() {
        $admin_user = get_user_by('login', 'admin');
        $passed = $admin_user === false;
        
        $this->add_check(
            'admin_username',
            'Kein "admin" Benutzer',
            $passed,
            5,
            'Standard-Benutzername vermeiden',
            $passed ? '' : 'Ändern Sie den Benutzernamen "admin" in einen anderen.'
        );
    }

    private function check_file_permissions() {
        $wp_config = ABSPATH . 'wp-config.php';
        $passed = true;
        
        if (file_exists($wp_config)) {
            $perms = substr(sprintf('%o', fileperms($wp_config)), -4);
            $passed = in_array($perms, ['0400', '0440', '0600', '0640', '0444']);
        }
        
        $this->add_check(
            'file_permissions',
            'wp-config.php geschützt',
            $passed,
            5,
            'Dateiberechtigungen korrekt gesetzt',
            $passed ? '' : 'Setzen Sie wp-config.php auf 0440 oder 0400.'
        );
    }

    private function check_plugins_updates() {
        $updates = get_site_transient('update_plugins');
        $count = !empty($updates->response) ? count($updates->response) : 0;
        $passed = $count === 0;
        
        $this->add_check(
            'plugins_updates',
            'Plugins aktuell',
            $passed,
            8,
            $count > 0 ? "{$count} Plugin-Update(s) verfügbar" : 'Alle Plugins aktuell',
            $passed ? '' : 'Aktualisieren Sie alle Plugins.'
        );
    }

    private function check_themes_updates() {
        $updates = get_site_transient('update_themes');
        $count = !empty($updates->response) ? count($updates->response) : 0;
        $passed = $count === 0;
        
        $this->add_check(
            'themes_updates',
            'Themes aktuell',
            $passed,
            5,
            $count > 0 ? "{$count} Theme-Update(s) verfügbar" : 'Alle Themes aktuell',
            $passed ? '' : 'Aktualisieren Sie alle Themes.'
        );
    }

    private function check_inactive_plugins() {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $all = get_plugins();
        $active = get_option('active_plugins', []);
        $inactive = count($all) - count($active);
        $passed = $inactive <= 2;
        
        $this->add_check(
            'inactive_plugins',
            'Inaktive Plugins entfernt',
            $passed,
            3,
            "{$inactive} inaktive Plugin(s)",
            $passed ? '' : 'Löschen Sie nicht verwendete Plugins.'
        );
    }

    private function check_login_url() {
        $custom = get_option('safetypress_custom_login_url', '');
        $enabled = get_option('safetypress_enable_custom_login', 0);
        $passed = !empty($custom) && $enabled;
        
        $this->add_check(
            'login_url',
            'Login-URL angepasst',
            $passed,
            5,
            'Versteckt die Standard Login-Seite',
            $passed ? '' : 'Ändern Sie die Login-URL in SafetyPress.'
        );
    }

    private function check_scheduled_scans() {
        $schedule = get_option('safetypress_scheduled_scan', 'disabled');
        $passed = $schedule !== 'disabled';
        
        $this->add_check(
            'scheduled_scans',
            'Geplante Scans aktiviert',
            $passed,
            5,
            'Automatische Malware-Scans',
            $passed ? '' : 'Aktivieren Sie tägliche oder wöchentliche Scans.'
        );
    }

    private function check_integrity() {
        $last_check = get_option('safetypress_last_integrity_check', []);
        $passed = isset($last_check['status']) && $last_check['status'] === 'ok';
        
        $this->add_check(
            'integrity',
            'Core-Integrität OK',
            $passed,
            8,
            'WordPress Core-Dateien unverändert',
            $passed ? '' : 'Prüfen und reparieren Sie modifizierte Core-Dateien.'
        );
    }

    private function check_backup() {
        // Prüfen ob gängige Backup-Plugins aktiv sind
        $backup_plugins = ['updraftplus', 'backwpup', 'duplicator', 'all-in-one-wp-migration'];
        $active = get_option('active_plugins', []);
        
        $has_backup = false;
        foreach ($active as $plugin) {
            foreach ($backup_plugins as $backup) {
                if (strpos($plugin, $backup) !== false) {
                    $has_backup = true;
                    break 2;
                }
            }
        }
        
        $this->add_check(
            'backup',
            'Backup-Plugin aktiv',
            $has_backup,
            5,
            'Regelmäßige Backups sind wichtig',
            $has_backup ? '' : 'Installieren Sie ein Backup-Plugin wie UpdraftPlus.'
        );
    }

    // ========================================
    // HILFSMETHODEN
    // ========================================

    /**
     * Note basierend auf Prozent
     */
    private function get_grade($percentage) {
        if ($percentage >= 90) return ['grade' => 'A', 'label' => 'Ausgezeichnet', 'color' => '#28a745'];
        if ($percentage >= 80) return ['grade' => 'B', 'label' => 'Gut', 'color' => '#5cb85c'];
        if ($percentage >= 70) return ['grade' => 'C', 'label' => 'Befriedigend', 'color' => '#f0ad4e'];
        if ($percentage >= 60) return ['grade' => 'D', 'label' => 'Ausreichend', 'color' => '#ff9800'];
        return ['grade' => 'F', 'label' => 'Mangelhaft', 'color' => '#dc3545'];
    }

    /**
     * Top-Empfehlungen holen
     */
    private function get_recommendations() {
        $recommendations = [];
        
        // Nach Gewicht sortieren (höchstes zuerst)
        $failed_checks = array_filter($this->checks, function($check) {
            return !$check['passed'] && !empty($check['recommendation']);
        });
        
        uasort($failed_checks, function($a, $b) {
            return $b['weight'] - $a['weight'];
        });
        
        // Top 5 Empfehlungen
        $count = 0;
        foreach ($failed_checks as $id => $check) {
            if ($count >= 5) break;
            
            $recommendations[] = [
                'id' => $id,
                'title' => $check['name'],
                'description' => $check['recommendation'],
                'impact' => $check['weight'] >= 10 ? 'high' : ($check['weight'] >= 5 ? 'medium' : 'low'),
            ];
            $count++;
        }
        
        return $recommendations;
    }

    /**
     * AJAX: Score abrufen
     */
    public function ajax_get_score() {
        check_ajax_referer('safetypress_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Keine Berechtigung']);
        }
        
        $result = $this->calculate_score();
        wp_send_json_success($result);
    }
}
