<?php
/**
 * SafetyPress Geplante Scans
 * 
 * Automatische Malware-Scans täglich/wöchentlich mit E-Mail-Benachrichtigung.
 */

if (!defined('ABSPATH')) exit;

class SafetyPress_Scheduled_Scanner {

    public function __construct() {
        // Cron-Hooks registrieren
        add_action('safetypress_daily_scan', [$this, 'run_scheduled_scan']);
        add_action('safetypress_weekly_scan', [$this, 'run_scheduled_scan']);
        
        // Schedule bei Einstellungsänderung aktualisieren
        add_action('update_option_safetypress_scheduled_scan', [$this, 'reschedule_scan'], 10, 2);
        add_action('update_option_safetypress_scheduled_scan_time', [$this, 'reschedule_on_time_change'], 10, 2);
        
        // Bei Aktivierung planen
        add_action('admin_init', [$this, 'maybe_schedule']);
        
        // AJAX für manuellen Scan
        add_action('wp_ajax_safetypress_run_scheduled_scan', [$this, 'ajax_run_scan']);
    }

    /**
     * Prüft und plant Scans wenn nötig
     */
    public function maybe_schedule() {
        $schedule = get_option('safetypress_scheduled_scan', 'disabled');
        
        if ($schedule === 'disabled') {
            $this->clear_schedules();
            return;
        }
        
        $scan_time = get_option('safetypress_scheduled_scan_time', '03:00');
        $time_parts = explode(':', $scan_time);
        $hour = isset($time_parts[0]) ? intval($time_parts[0]) : 3;
        $minute = isset($time_parts[1]) ? intval($time_parts[1]) : 0;
        
        if ($schedule === 'daily' && !wp_next_scheduled('safetypress_daily_scan')) {
            $next_run = strtotime("today {$hour}:{$minute}");
            if ($next_run < time()) {
                $next_run = strtotime("tomorrow {$hour}:{$minute}");
            }
            wp_schedule_event($next_run, 'daily', 'safetypress_daily_scan');
        }
        
        if ($schedule === 'weekly' && !wp_next_scheduled('safetypress_weekly_scan')) {
            $next_run = strtotime("next monday {$hour}:{$minute}");
            wp_schedule_event($next_run, 'weekly', 'safetypress_weekly_scan');
        }
    }

    /**
     * Schedule bei Änderung neu planen
     */
    public function reschedule_scan($old_value, $new_value) {
        $this->clear_schedules();
        
        $scan_time = get_option('safetypress_scheduled_scan_time', '03:00');
        $time_parts = explode(':', $scan_time);
        $hour = isset($time_parts[0]) ? intval($time_parts[0]) : 3;
        $minute = isset($time_parts[1]) ? intval($time_parts[1]) : 0;
        
        if ($new_value === 'daily') {
            $next_run = strtotime("tomorrow {$hour}:{$minute}");
            wp_schedule_event($next_run, 'daily', 'safetypress_daily_scan');
        } elseif ($new_value === 'weekly') {
            $next_run = strtotime("next monday {$hour}:{$minute}");
            wp_schedule_event($next_run, 'weekly', 'safetypress_weekly_scan');
        }
    }
    
    /**
     * Bei Zeit-Änderung Schedule neu planen
     */
    public function reschedule_on_time_change($old_value, $new_value) {
        $schedule = get_option('safetypress_scheduled_scan', 'disabled');
        if ($schedule !== 'disabled') {
            $this->reschedule_scan('', $schedule);
        }
    }

    /**
     * Alle Schedules löschen
     */
    private function clear_schedules() {
        wp_clear_scheduled_hook('safetypress_daily_scan');
        wp_clear_scheduled_hook('safetypress_weekly_scan');
    }

    /**
     * Führt den geplanten Scan aus
     */
    public function run_scheduled_scan() {
        // Nur wenn Pro aktiv
        if (!function_exists('safetypress') || !safetypress()->is_pro()) {
            return;
        }
        
        // Prüfen ob E-Mail-Benachrichtigung aktiviert
        if (!get_option('safetypress_scheduled_scan_email', 1)) {
            return;
        }
        
        // Scan ausführen
        $results = $this->perform_deep_scan();
        
        // Ergebnisse speichern
        update_option('safetypress_last_scheduled_scan', [
            'time' => current_time('mysql'),
            'results' => $results,
        ]);
        
        // E-Mail senden
        do_action('safetypress_scheduled_scan_completed', $results);
        
        // Log
        if (function_exists('safetypress') && safetypress()->logger) {
            $issue_count = count($results['issues'] ?? []);
            safetypress()->logger->log('scheduled_scan', "Geplanter Scan abgeschlossen: {$issue_count} Problem(e)", [
                'issues_found' => $issue_count,
                'files_scanned' => $results['stats']['files_scanned'] ?? 0,
            ]);
        }
    }

    /**
     * Tiefenscan ausführen (scannt ALLE Dateien)
     */
    private function perform_deep_scan() {
        $start_time = microtime(true);
        
        $results = [
            'issues' => [],
            'stats' => [
                'files_scanned' => 0,
                'directories_scanned' => 0,
            ],
            'scan_type' => 'scheduled_deep',
            'scan_time' => current_time('mysql'),
        ];
        
        // Patterns von API laden
        $patterns = $this->get_scan_patterns();
        
        // NUR wp-content scannen für Malware (nicht Core-Dateien!)
        // Core-Dateien werden separat durch Integritätsprüfung geprüft
        $scan_dirs = [
            ABSPATH . 'wp-content/plugins/',
            ABSPATH . 'wp-content/themes/',
            ABSPATH . 'wp-content/uploads/',
        ];
        
        foreach ($scan_dirs as $dir) {
            if (!is_dir($dir)) continue;
            $scan_result = $this->scan_directory($dir, $patterns, 5);
            $results['issues'] = array_merge($results['issues'], $scan_result['issues']);
            $results['stats']['files_scanned'] += $scan_result['files'];
            $results['stats']['directories_scanned'] += $scan_result['dirs'];
        }
        
        // Root-Dateien prüfen (nur unbekannte Dateien, keine Core-Dateien)
        $root_result = $this->scan_root_files($patterns);
        $results['issues'] = array_merge($results['issues'], $root_result);
        
        // Core-Integrität prüfen (als Integritäts-Problem, NICHT als Malware)
        $integrity_result = $this->check_core_integrity();
        $results['issues'] = array_merge($results['issues'], $integrity_result);
        
        // Dauer berechnen
        $results['stats']['duration'] = round(microtime(true) - $start_time, 2);
        
        return $results;
    }

    /**
     * Verzeichnis rekursiv scannen
     */
    private function scan_directory($dir, $patterns, $max_depth = 3, $current_depth = 0) {
        $result = [
            'issues' => [],
            'files' => 0,
            'dirs' => 1,
        ];
        
        if ($current_depth > $max_depth) {
            return $result;
        }
        
        $items = @scandir($dir);
        if (!$items) return $result;
        
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') continue;
            
            $path = $dir . DIRECTORY_SEPARATOR . $item;
            
            // Bestimmte Verzeichnisse überspringen
            $skip_dirs = ['node_modules', 'vendor', '.git', 'cache', 'backups'];
            if (is_dir($path) && in_array($item, $skip_dirs)) continue;
            
            // WordPress Core-Verzeichnisse überspringen (werden durch Integritätsprüfung geprüft)
            if (is_dir($path) && in_array($item, ['wp-admin', 'wp-includes'])) continue;
            
            if (is_dir($path)) {
                $sub_result = $this->scan_directory($path, $patterns, $max_depth, $current_depth + 1);
                $result['issues'] = array_merge($result['issues'], $sub_result['issues']);
                $result['files'] += $sub_result['files'];
                $result['dirs'] += $sub_result['dirs'];
            } elseif (is_file($path)) {
                // Nur PHP und JS Dateien scannen
                $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
                if (!in_array($ext, ['php', 'js', 'html', 'htm'])) continue;
                
                // Große Dateien überspringen (>1MB)
                if (filesize($path) > 1048576) continue;
                
                $result['files']++;
                
                // SafetyPress selbst überspringen
                if (strpos($path, 'safetypress') !== false) continue;
                
                // WordPress Core-Dateien überspringen (wp-*.php im Root)
                $relative_path = str_replace(ABSPATH, '', $path);
                if (preg_match('/^wp-[a-z-]+\.php$/', $relative_path)) continue;
                
                $content = @file_get_contents($path);
                if (!$content) continue;
                
                foreach ($patterns as $key => $pattern_data) {
                    if (!isset($pattern_data['pattern'])) continue;
                    
                    if (@preg_match($pattern_data['pattern'], $content)) {
                        $result['issues'][] = [
                            'type' => $pattern_data['severity'] ?? 'warning',
                            'file' => str_replace(ABSPATH, '', $path),
                            'message' => 'Verdächtig: ' . ($pattern_data['name'] ?? $key),
                            'recommendation' => $pattern_data['description'] ?? 'Überprüfen Sie diese Datei.',
                        ];
                        
                        // Malware-Fund melden (nur für NICHT-Core-Dateien)
                        if (($pattern_data['severity'] ?? '') === 'critical') {
                            do_action('safetypress_malware_found', str_replace(ABSPATH, '', $path), $pattern_data['name'] ?? $key);
                        }
                        
                        break; // Nur erste Übereinstimmung pro Datei
                    }
                }
            }
        }
        
        return $result;
    }

    /**
     * Root-Dateien prüfen
     */
    private function scan_root_files($patterns) {
        $issues = [];
        
        $root_files = glob(ABSPATH . '*.php');
        $core_files = ['wp-config.php', 'wp-settings.php', 'wp-load.php', 'wp-blog-header.php', 
                      'wp-cron.php', 'wp-links-opml.php', 'wp-login.php', 'wp-mail.php', 
                      'wp-signup.php', 'wp-trackback.php', 'wp-activate.php', 
                      'wp-comments-post.php', 'index.php', 'xmlrpc.php'];
        
        foreach ($root_files as $file) {
            $filename = basename($file);
            
            // Unbekannte PHP-Dateien im Root sind verdächtig
            if (!in_array($filename, $core_files)) {
                $issues[] = [
                    'type' => 'warning',
                    'file' => $filename,
                    'message' => 'Unbekannte PHP-Datei im Root-Verzeichnis',
                    'recommendation' => 'Diese Datei gehört nicht zu WordPress. Prüfen Sie ob sie legitim ist.',
                ];
            }
        }
        
        return $issues;
    }

    /**
     * WordPress Core-Integrität prüfen
     * HINWEIS: Modifizierte Core-Dateien sind KEINE Malware, sondern Integritäts-Probleme
     */
    private function check_core_integrity() {
        $issues = [];
        
        global $wp_version;
        
        // Checksums von WordPress.org holen
        $response = wp_remote_get("https://api.wordpress.org/core/checksums/1.0/?version={$wp_version}&locale=de_DE");
        
        if (is_wp_error($response)) {
            return $issues;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (empty($data['checksums'])) {
            return $issues;
        }
        
        // Kritische Core-Dateien prüfen
        $critical_files = [
            'wp-login.php',
            'wp-includes/version.php',
            'wp-includes/pluggable.php',
            'wp-admin/admin.php',
            'wp-admin/includes/class-wp-upgrader.php',
        ];
        
        foreach ($critical_files as $file) {
            if (!isset($data['checksums'][$file])) continue;
            
            $path = ABSPATH . $file;
            if (!file_exists($path)) continue;
            
            $local_hash = md5_file($path);
            $remote_hash = $data['checksums'][$file];
            
            if ($local_hash !== $remote_hash) {
                // Als "integrity" statt "critical" markieren - das ist KEINE Malware!
                $issues[] = [
                    'type' => 'integrity',
                    'file' => $file,
                    'message' => 'Core-Datei modifiziert (Integritätsprüfung)',
                    'recommendation' => 'Diese Datei unterscheidet sich vom Original. Nutzen Sie die Integritätsprüfung um sie zu reparieren.',
                ];
            }
        }
        
        return $issues;
    }

    /**
     * Scan-Patterns von API laden
     */
    private function get_scan_patterns() {
        $cache_key = 'safetypress_scan_patterns';
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        $response = wp_remote_get('https://safetypress.de/api/license.php?action=get_scan_patterns', ['timeout' => 15]);
        
        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return $this->get_fallback_patterns();
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (empty($data['patterns'])) {
            return $this->get_fallback_patterns();
        }
        
        set_transient($cache_key, $data['patterns'], DAY_IN_SECONDS);
        
        return $data['patterns'];
    }

    /**
     * Fallback-Patterns
     */
    private function get_fallback_patterns() {
        return [
            'long_encoded' => [
                'pattern' => '/[\'"][A-Za-z0-9+\/=]{500,}[\'"]/i',
                'severity' => 'warning',
                'name' => 'Langer encodierter String',
                'description' => 'Möglicherweise versteckter Code',
            ],
        ];
    }

    /**
     * AJAX: Manueller Scan
     */
    public function ajax_run_scan() {
        check_ajax_referer('safetypress_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Keine Berechtigung']);
        }
        
        $results = $this->perform_deep_scan();
        
        update_option('safetypress_last_scheduled_scan', [
            'time' => current_time('mysql'),
            'results' => $results,
        ]);
        
        wp_send_json_success($results);
    }

    /**
     * Status des geplanten Scans
     */
    public function get_status() {
        $schedule = get_option('safetypress_scheduled_scan', 'disabled');
        $last_scan = get_option('safetypress_last_scheduled_scan', []);
        
        $next_daily = wp_next_scheduled('safetypress_daily_scan');
        $next_weekly = wp_next_scheduled('safetypress_weekly_scan');
        
        return [
            'schedule' => $schedule,
            'last_scan' => $last_scan['time'] ?? null,
            'last_issues' => count($last_scan['results']['issues'] ?? []),
            'next_scan' => $next_daily ?: $next_weekly,
        ];
    }
}
