<?php
/**
 * SafetyPress Security Scanner
 * 
 * Scannt WordPress auf Sicherheitsprobleme.
 * Signaturen werden von der API geladen um False-Positives bei Hostern zu vermeiden.
 */

if (!defined('ABSPATH')) exit;

class SafetyPress_Scanner {

    private $stats = array();
    private $api_url = 'https://safetypress.de/api/license.php';

    public function __construct() {
        add_action('wp_ajax_safetypress_run_scan', array($this, 'ajax_run_scan'));
        $this->reset_stats();
    }

    private function reset_stats() {
        $this->stats = array(
            'files_scanned' => 0,
            'directories_scanned' => 0,
            'checks_performed' => 0,
            'scanned_areas' => array(),
            'scan_details' => array(),
        );
    }

    public function quick_scan() {
        $this->reset_stats();
        $start_time = microtime(true);
        
        $results = array(
            'issues' => array(),
            'scan_type' => 'quick',
            'scan_time' => current_time('mysql'),
        );
        
        // Core-Dateien prüfen
        $core_result = $this->check_core();
        $results['issues'] = array_merge($results['issues'], $core_result);
        
        // Uploads prüfen
        $uploads_result = $this->check_uploads();
        $results['issues'] = array_merge($results['issues'], $uploads_result);
        
        // Berechtigungen prüfen
        $perms_result = $this->check_permissions();
        $results['issues'] = array_merge($results['issues'], $perms_result);
        
        // Themes prüfen
        $themes_result = $this->check_themes();
        $results['issues'] = array_merge($results['issues'], $themes_result);
        
        // Plugins prüfen (Basis)
        $plugins_result = $this->check_plugins_basic();
        $results['issues'] = array_merge($results['issues'], $plugins_result);
        
        // .htaccess prüfen
        $htaccess_result = $this->check_htaccess();
        $results['issues'] = array_merge($results['issues'], $htaccess_result);
        
        // Statistiken hinzufügen
        $results['stats'] = $this->stats;
        $results['stats']['duration'] = round(microtime(true) - $start_time, 2);
        
        return $results;
    }

    public function deep_scan() {
        if (!function_exists('safetypress') || !safetypress()->is_pro()) {
            return array('error' => true, 'message' => 'Deep-Scan ist Pro.');
        }
        
        $results = $this->quick_scan();
        $results['scan_type'] = 'deep';
        
        // Tiefenscan mit API-Patterns
        $threat_result = $this->scan_for_threats();
        $results['issues'] = array_merge($results['issues'], $threat_result);
        
        // WordPress Core Integrität
        $integrity_result = $this->check_core_integrity();
        $results['issues'] = array_merge($results['issues'], $integrity_result);
        
        // Statistiken aktualisieren
        $results['stats'] = $this->stats;
        
        return $results;
    }

    private function check_core() {
        $issues = array();
        $this->stats['scanned_areas'][] = 'WordPress Core-Konfiguration';
        $this->stats['checks_performed']++;
        
        $details = array(
            'area' => 'Core-Konfiguration',
            'icon' => '⚙️',
            'items' => array()
        );
        
        $config = ABSPATH . 'wp-config.php';
        if (file_exists($config)) {
            $this->stats['files_scanned']++;
            $content = @file_get_contents($config);
            
            $details['items'][] = array(
                'name' => 'wp-config.php',
                'status' => 'checked',
                'detail' => 'Konfigurationsdatei analysiert'
            );
            
            if ($content && preg_match("/WP_DEBUG['\"]\s*,\s*true/i", $content)) {
                $issues[] = array(
                    'type' => 'warning', 
                    'file' => 'wp-config.php', 
                    'message' => 'Debug-Modus ist aktiviert.',
                    'recommendation' => 'Deaktivieren Sie WP_DEBUG auf Produktionsseiten.'
                );
                $details['items'][0]['status'] = 'warning';
            }
            
            if ($content && preg_match("/table_prefix\s*=\s*['\"]wp_['\"]/i", $content)) {
                $issues[] = array(
                    'type' => 'info', 
                    'file' => 'wp-config.php', 
                    'message' => 'Standard Datenbank-Präfix (wp_) wird verwendet.',
                    'recommendation' => 'Ein benutzerdefiniertes Präfix erhöht die Sicherheit.'
                );
            }
            $this->stats['checks_performed']++;
            
            if ($content && !preg_match("/define\s*\(\s*['\"]AUTH_KEY['\"]\s*,\s*['\"].{32,}['\"]\s*\)/i", $content)) {
                $issues[] = array(
                    'type' => 'critical', 
                    'file' => 'wp-config.php', 
                    'message' => 'Security Keys fehlen oder sind zu kurz.',
                    'recommendation' => 'Generieren Sie neue Keys auf api.wordpress.org/secret-key'
                );
            }
            $this->stats['checks_performed']++;
        }
        
        global $wp_version;
        $details['items'][] = array(
            'name' => 'WordPress Version',
            'status' => 'checked',
            'detail' => 'Version ' . $wp_version
        );
        $this->stats['checks_performed']++;
        
        $this->stats['scan_details'][] = $details;
        
        return $issues;
    }

    private function check_uploads() {
        $issues = array();
        $this->stats['scanned_areas'][] = 'Upload-Verzeichnis';
        
        $details = array(
            'area' => 'Uploads',
            'icon' => '📁',
            'items' => array()
        );
        
        $upload_dir = wp_upload_dir();
        $base = $upload_dir['basedir'];
        
        if (is_dir($base)) {
            $this->stats['directories_scanned']++;
            
            $php_files = glob($base . '/*.php');
            $dangerous_files = 0;
            
            if (!empty($php_files)) {
                foreach ($php_files as $file) {
                    $this->stats['files_scanned']++;
                    $filename = basename($file);
                    
                    if ($filename === 'index.php') {
                        $content = @file_get_contents($file);
                        if ($content !== false) {
                            $content_lower = strtolower(trim($content));
                            if (strpos($content_lower, 'silence is golden') !== false || 
                                strlen($content) < 50) {
                                continue;
                            }
                        }
                    }
                    
                    $dangerous_files++;
                    $issues[] = array(
                        'type' => 'critical',
                        'file' => 'uploads/' . $filename,
                        'message' => 'PHP-Datei im Upload-Verzeichnis gefunden.',
                        'recommendation' => 'PHP-Dateien gehören nicht in den Upload-Ordner.'
                    );
                }
            }
            
            $details['items'][] = array(
                'name' => 'Upload-Ordner',
                'status' => $dangerous_files > 0 ? 'critical' : 'ok',
                'detail' => $dangerous_files > 0 
                    ? sprintf('%d verdächtige PHP-Dateien', $dangerous_files)
                    : 'Keine PHP-Dateien gefunden'
            );
        }
        
        $htaccess = $base . '/.htaccess';
        if (!file_exists($htaccess)) {
            $issues[] = array(
                'type' => 'info',
                'file' => 'uploads/.htaccess',
                'message' => 'Keine .htaccess im Upload-Verzeichnis.',
                'recommendation' => 'PHP-Ausführung sollte hier blockiert werden.'
            );
        }
        $this->stats['checks_performed']++;
        
        $this->stats['scan_details'][] = $details;
        
        return $issues;
    }

    private function check_permissions() {
        $issues = array();
        $this->stats['scanned_areas'][] = 'Dateiberechtigungen';
        
        $details = array(
            'area' => 'Berechtigungen',
            'icon' => '🔐',
            'items' => array()
        );
        
        $critical_files = array(
            'wp-config.php' => '0440',
            '.htaccess' => '0644',
        );
        
        foreach ($critical_files as $file => $rec_perms) {
            $path = ABSPATH . $file;
            if (file_exists($path)) {
                $this->stats['files_scanned']++;
                $perms = substr(sprintf('%o', fileperms($path)), -4);
                $writable = is_writable($path);
                
                $status = 'ok';
                if ($file === 'wp-config.php' && $writable) {
                    $status = 'warning';
                    $issues[] = array(
                        'type' => 'warning',
                        'file' => $file,
                        'message' => sprintf('%s ist schreibbar (Perms: %s).', $file, $perms),
                        'recommendation' => 'Empfohlen: ' . $rec_perms
                    );
                }
                
                $details['items'][] = array(
                    'name' => $file,
                    'status' => $status,
                    'detail' => 'Perms: ' . $perms
                );
            }
            $this->stats['checks_performed']++;
        }
        
        $this->stats['scan_details'][] = $details;
        
        return $issues;
    }

    private function check_themes() {
        $issues = array();
        $this->stats['scanned_areas'][] = 'Themes';
        
        $details = array(
            'area' => 'Themes',
            'icon' => '🎨',
            'items' => array()
        );
        
        $themes = wp_get_themes();
        $active_theme = wp_get_theme();
        $inactive_count = 0;
        
        foreach ($themes as $slug => $theme) {
            $this->stats['checks_performed']++;
            if ($slug !== $active_theme->get_stylesheet() && $slug !== $active_theme->get_template()) {
                $inactive_count++;
            }
        }
        
        if ($inactive_count > 1) {
            $issues[] = array(
                'type' => 'info',
                'file' => 'wp-content/themes/',
                'message' => sprintf('%d inaktive Themes vorhanden.', $inactive_count),
                'recommendation' => 'Entfernen Sie nicht verwendete Themes.'
            );
        }
        
        $details['items'][] = array(
            'name' => 'Aktives Theme',
            'status' => 'ok',
            'detail' => $active_theme->get('Name') . ' v' . $active_theme->get('Version')
        );
        
        $details['items'][] = array(
            'name' => 'Inaktive Themes',
            'status' => $inactive_count > 2 ? 'info' : 'ok',
            'detail' => $inactive_count . ' gefunden'
        );
        
        $this->stats['scan_details'][] = $details;
        
        return $issues;
    }

    private function check_plugins_basic() {
        $issues = array();
        $this->stats['scanned_areas'][] = 'Plugins';
        
        $details = array(
            'area' => 'Plugins',
            'icon' => '🔌',
            'items' => array()
        );
        
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $all_plugins = get_plugins();
        $active_plugins = get_option('active_plugins', array());
        $inactive_count = 0;
        $update_count = 0;
        
        foreach ($all_plugins as $plugin_file => $plugin_data) {
            $this->stats['checks_performed']++;
            
            if (!in_array($plugin_file, $active_plugins)) {
                $inactive_count++;
            }
        }
        
        $updates = get_site_transient('update_plugins');
        if ($updates && !empty($updates->response)) {
            $update_count = count($updates->response);
            if ($update_count > 0) {
                $issues[] = array(
                    'type' => 'warning',
                    'file' => 'wp-content/plugins/',
                    'message' => sprintf('%d Plugin-Update(s) verfügbar.', $update_count),
                    'recommendation' => 'Halten Sie Ihre Plugins aktuell.'
                );
            }
        }
        
        if ($inactive_count > 3) {
            $issues[] = array(
                'type' => 'info',
                'file' => 'wp-content/plugins/',
                'message' => sprintf('%d inaktive Plugins vorhanden.', $inactive_count),
                'recommendation' => 'Entfernen Sie nicht verwendete Plugins.'
            );
        }
        
        $details['items'][] = array(
            'name' => 'Aktive Plugins',
            'status' => 'ok',
            'detail' => count($active_plugins) . ' aktiv'
        );
        
        $details['items'][] = array(
            'name' => 'Inaktive Plugins',
            'status' => $inactive_count > 3 ? 'info' : 'ok',
            'detail' => $inactive_count . ' gefunden'
        );
        
        $details['items'][] = array(
            'name' => 'Updates',
            'status' => $update_count > 0 ? 'warning' : 'ok',
            'detail' => $update_count > 0 ? $update_count . ' verfügbar' : 'Alle aktuell'
        );
        
        $this->stats['scan_details'][] = $details;
        
        return $issues;
    }

    private function check_htaccess() {
        $issues = array();
        $this->stats['scanned_areas'][] = '.htaccess';
        
        $details = array(
            'area' => '.htaccess',
            'icon' => '📄',
            'items' => array()
        );
        
        $htaccess = ABSPATH . '.htaccess';
        if (file_exists($htaccess)) {
            $this->stats['files_scanned']++;
            $content = @file_get_contents($htaccess);
            
            if ($content) {
                $details['items'][] = array(
                    'name' => '.htaccess',
                    'status' => 'ok',
                    'detail' => 'Datei vorhanden (' . strlen($content) . ' Bytes)'
                );
            }
        } else {
            $details['items'][] = array(
                'name' => '.htaccess',
                'status' => 'info',
                'detail' => 'Nicht vorhanden'
            );
        }
        
        $this->stats['checks_performed']++;
        $this->stats['scan_details'][] = $details;
        
        return $issues;
    }

    /**
     * Lädt Scan-Patterns von der API
     */
    private function get_scan_patterns() {
        $cache_key = 'safetypress_scan_patterns';
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        $response = wp_remote_get(
            $this->api_url . '?action=get_scan_patterns',
            array('timeout' => 15)
        );
        
        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return $this->get_basic_patterns();
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (empty($data) || !isset($data['patterns'])) {
            return $this->get_basic_patterns();
        }
        
        set_transient($cache_key, $data['patterns'], DAY_IN_SECONDS);
        
        return $data['patterns'];
    }

    /**
     * Basis-Patterns (Fallback)
     */
    private function get_basic_patterns() {
        // Nur minimale, unverdächtige Patterns
        return array(
            'long_encoded_string' => array(
                'pattern' => '/[\'"][A-Za-z0-9+\/=]{500,}[\'"]/i',
                'severity' => 'warning',
                'name' => 'Langer encodierter String',
                'description' => 'Sehr langer encodierter String gefunden'
            ),
        );
    }

    /**
     * Tiefenscan mit API-Patterns
     */
    private function scan_for_threats() {
        $issues = array();
        $this->stats['scanned_areas'][] = 'Tiefenscan (Pro)';
        
        $details = array(
            'area' => 'Tiefenscan',
            'icon' => '🔍',
            'items' => array()
        );
        
        $patterns = $this->get_scan_patterns();
        
        if (empty($patterns)) {
            $details['items'][] = array(
                'name' => 'Pattern-Status',
                'status' => 'warning',
                'detail' => 'Signaturen konnten nicht geladen werden'
            );
            $this->stats['scan_details'][] = $details;
            return $issues;
        }
        
        $scanned_files = 0;
        $suspicious_files = 0;
        $found_signatures = array();
        
        // Plugins scannen
        $plugin_dir = ABSPATH . 'wp-content/plugins/';
        if (is_dir($plugin_dir)) {
            $files = $this->get_php_files_recursive($plugin_dir, 3);
            foreach ($files as $file) {
                if (strpos($file, 'safetypress') !== false) continue;
                
                $this->stats['files_scanned']++;
                $scanned_files++;
                $content = @file_get_contents($file);
                if (!$content) continue;
                
                foreach ($patterns as $key => $pattern_data) {
                    if (!isset($pattern_data['pattern'])) continue;
                    
                    if (@preg_match($pattern_data['pattern'], $content)) {
                        $suspicious_files++;
                        $found_signatures[$key] = ($found_signatures[$key] ?? 0) + 1;
                        
                        $issues[] = array(
                            'type' => $pattern_data['severity'] ?? 'warning',
                            'file' => str_replace(ABSPATH, '', $file),
                            'message' => 'Verdächtig: ' . ($pattern_data['name'] ?? 'Unbekannt'),
                            'recommendation' => $pattern_data['description'] ?? 'Überprüfen Sie diese Datei manuell.'
                        );
                        break;
                    }
                }
            }
        }
        
        // Themes scannen
        $theme_dir = ABSPATH . 'wp-content/themes/';
        if (is_dir($theme_dir)) {
            $files = $this->get_php_files_recursive($theme_dir, 3);
            foreach ($files as $file) {
                $this->stats['files_scanned']++;
                $scanned_files++;
                $content = @file_get_contents($file);
                if (!$content) continue;
                
                foreach ($patterns as $key => $pattern_data) {
                    if (!isset($pattern_data['pattern'])) continue;
                    
                    if (@preg_match($pattern_data['pattern'], $content)) {
                        $suspicious_files++;
                        $found_signatures[$key] = ($found_signatures[$key] ?? 0) + 1;
                        
                        $issues[] = array(
                            'type' => $pattern_data['severity'] ?? 'warning',
                            'file' => str_replace(ABSPATH, '', $file),
                            'message' => 'Verdächtig: ' . ($pattern_data['name'] ?? 'Unbekannt'),
                            'recommendation' => $pattern_data['description'] ?? 'Überprüfen Sie diese Datei manuell.'
                        );
                        break;
                    }
                }
            }
        }
        
        $details['items'][] = array(
            'name' => 'Dateien gescannt',
            'status' => $suspicious_files > 0 ? 'warning' : 'ok',
            'detail' => sprintf('%d Dateien, %d verdächtig', $scanned_files, $suspicious_files)
        );
        
        $details['items'][] = array(
            'name' => 'Signaturen',
            'status' => 'checked',
            'detail' => sprintf('%d Muster geprüft', count($patterns))
        );
        
        $this->stats['checks_performed'] += count($patterns);
        $this->stats['scan_details'][] = $details;
        
        return $issues;
    }

    private function check_core_integrity() {
        $issues = array();
        $this->stats['scanned_areas'][] = 'Core-Integrität';
        
        $details = array(
            'area' => 'Core-Integrität',
            'icon' => '✓',
            'items' => array()
        );
        
        $core_files = array(
            'wp-login.php',
            'wp-includes/version.php',
            'wp-includes/functions.php',
            'wp-admin/admin.php'
        );
        
        $checked = 0;
        
        foreach ($core_files as $file) {
            $path = ABSPATH . $file;
            if (file_exists($path)) {
                $this->stats['files_scanned']++;
                $checked++;
            }
        }
        
        $details['items'][] = array(
            'name' => 'Core-Dateien',
            'status' => 'ok',
            'detail' => sprintf('%d geprüft', $checked)
        );
        
        $this->stats['checks_performed']++;
        $this->stats['scan_details'][] = $details;
        
        return $issues;
    }

    private function get_php_files_recursive($dir, $depth = 2) {
        $files = array();
        if ($depth < 0 || !is_dir($dir)) return $files;
        
        $this->stats['directories_scanned']++;
        
        $items = @scandir($dir);
        if (!$items) return $files;
        
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') continue;
            
            $path = $dir . DIRECTORY_SEPARATOR . $item;
            
            if (is_file($path) && pathinfo($path, PATHINFO_EXTENSION) === 'php') {
                $files[] = $path;
            } elseif (is_dir($path)) {
                $files = array_merge($files, $this->get_php_files_recursive($path, $depth - 1));
            }
        }
        
        return $files;
    }

    public function ajax_run_scan() {
        check_ajax_referer('safetypress_admin', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        }
        
        $type = isset($_POST['scan_type']) ? sanitize_text_field($_POST['scan_type']) : 'quick';
        $results = ($type === 'deep') ? $this->deep_scan() : $this->quick_scan();
        set_transient('safetypress_last_scan', $results, DAY_IN_SECONDS);
        
        if (function_exists('safetypress') && safetypress()->logger) {
            $issue_count = isset($results['issues']) ? count($results['issues']) : 0;
            safetypress()->logger->log('scan_completed', sprintf('%s abgeschlossen: %d Problem(e) gefunden', ucfirst($type) . '-Scan', $issue_count), array(
                'scan_type' => $type,
                'issues_found' => $issue_count,
                'files_scanned' => $results['stats']['files_scanned'] ?? 0,
            ));
        }
        
        wp_send_json_success($results);
    }
}
