<?php
/**
 * SafetyPress E-Mail-Benachrichtigungen
 */

if (!defined('ABSPATH')) exit;

class SafetyPress_Notifications {

    private $from_email;
    private $from_name;

    public function __construct() {
        $this->from_email = get_option('safetypress_notification_from_email', get_option('admin_email'));
        $this->from_name = get_option('safetypress_notification_from_name', 'SafetyPress');
        
        // Hooks für Benachrichtigungen
        add_action('safetypress_brute_force_detected', [$this, 'notify_brute_force'], 10, 2);
        add_action('safetypress_malware_found', [$this, 'notify_malware'], 10, 2);
        add_action('safetypress_new_admin_login', [$this, 'notify_new_admin_login'], 10, 3);
        add_action('safetypress_license_expiring', [$this, 'notify_license_expiring'], 10, 2);
        add_action('safetypress_scheduled_scan_completed', [$this, 'notify_scan_results'], 10, 1);
        add_action('safetypress_integrity_issues_found', [$this, 'notify_integrity_issues'], 10, 1);
        
        // Admin Login tracken
        add_action('wp_login', [$this, 'track_admin_login'], 10, 2);
    }

    /**
     * Prüft ob Benachrichtigungen aktiviert sind
     */
    private function is_enabled($type = 'all') {
        if (!get_option('safetypress_email_notifications', 1)) {
            return false;
        }
        return true;
    }

    /**
     * Holt die Empfänger E-Mail(s)
     */
    private function get_recipients() {
        $emails = get_option('safetypress_notification_email', get_option('admin_email'));
        if (is_string($emails)) {
            $emails = array_map('trim', explode(',', $emails));
        }
        return array_filter($emails, 'is_email');
    }

    /**
     * Sendet eine E-Mail
     */
    public function send_email($subject, $message, $type = 'all') {
        if (!$this->is_enabled($type)) {
            return false;
        }
        
        $recipients = $this->get_recipients();
        if (empty($recipients)) {
            return false;
        }
        
        // Rate Limiting
        $rate_key = 'safetypress_email_rate_' . $type;
        $rate_count = get_transient($rate_key) ?: 0;
        if ($rate_count >= 10) {
            return false;
        }
        set_transient($rate_key, $rate_count + 1, HOUR_IN_SECONDS);
        
        $site_name = get_bloginfo('name');
        $site_url = get_site_url();
        $html_message = $this->get_email_template($subject, $message, $site_name, $site_url);
        
        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $this->from_name . ' <' . $this->from_email . '>',
        ];
        
        foreach ($recipients as $recipient) {
            wp_mail($recipient, '[SafetyPress] ' . $subject, $html_message, $headers);
        }
        
        return true;
    }

    /**
     * E-Mail Template
     */
    private function get_email_template($subject, $content, $site_name, $site_url) {
        return '
        <!DOCTYPE html>
        <html>
        <head><meta charset="UTF-8"></head>
        <body style="margin:0;padding:0;background-color:#f4f4f4;font-family:Arial,sans-serif;">
            <table width="100%" cellpadding="0" cellspacing="0" style="background-color:#f4f4f4;padding:20px;">
                <tr><td align="center">
                    <table width="600" cellpadding="0" cellspacing="0" style="background-color:#ffffff;border-radius:8px;overflow:hidden;">
                        <!-- Header -->
                        <tr><td style="background:linear-gradient(135deg,#667eea 0%,#764ba2 100%);padding:30px;text-align:center;">
                            <h1 style="color:#ffffff;margin:0;font-size:24px;">🛡️ SafetyPress</h1>
                            <p style="color:rgba(255,255,255,0.8);margin:5px 0 0;font-size:14px;">Sicherheitsbenachrichtigung</p>
                        </td></tr>
                        
                        <!-- Content -->
                        <tr><td style="padding:30px;">
                            <h2 style="color:#1a1a2e;margin:0 0 20px;font-size:20px;">' . esc_html($subject) . '</h2>
                            <div style="color:#4a4a4a;font-size:15px;line-height:1.6;">' . $content . '</div>
                        </td></tr>
                        
                        <!-- Info -->
                        <tr><td style="padding:0 30px 30px;">
                            <table width="100%" style="background-color:#f8f9fa;border-radius:6px;padding:15px;">
                                <tr><td style="color:#666;font-size:13px;">
                                    <strong>Website:</strong> ' . esc_html($site_name) . '<br>
                                    <strong>URL:</strong> ' . esc_html($site_url) . '<br>
                                    <strong>Zeit:</strong> ' . current_time('d.m.Y H:i:s') . '
                                </td></tr>
                            </table>
                        </td></tr>
                        
                        <!-- Button -->
                        <tr><td style="padding:0 30px 30px;text-align:center;">
                            <a href="' . esc_url(admin_url('admin.php?page=safetypress')) . '" style="display:inline-block;background:linear-gradient(135deg,#667eea 0%,#764ba2 100%);color:#ffffff;text-decoration:none;padding:12px 30px;border-radius:6px;font-weight:600;">Dashboard öffnen</a>
                        </td></tr>
                        
                        <!-- Footer -->
                        <tr><td style="background-color:#f8f9fa;padding:20px;text-align:center;border-top:1px solid #eee;">
                            <p style="color:#999;font-size:12px;margin:0;">Diese E-Mail wurde automatisch von SafetyPress gesendet.</p>
                        </td></tr>
                    </table>
                </td></tr>
            </table>
        </body>
        </html>';
    }

    /**
     * Benachrichtigung: Brute-Force Angriff
     */
    public function notify_brute_force($ip, $attempts) {
        $subject = 'Brute-Force Angriff erkannt';
        $message = '
            <p style="color:#dc3545;font-weight:600;">⚠️ Ein Brute-Force Angriff wurde erkannt und blockiert!</p>
            <table style="width:100%;border-collapse:collapse;margin:20px 0;">
                <tr>
                    <td style="padding:10px;border-bottom:1px solid #eee;color:#666;">IP-Adresse:</td>
                    <td style="padding:10px;border-bottom:1px solid #eee;"><strong>' . esc_html($ip) . '</strong></td>
                </tr>
                <tr>
                    <td style="padding:10px;border-bottom:1px solid #eee;color:#666;">Fehlgeschlagene Versuche:</td>
                    <td style="padding:10px;border-bottom:1px solid #eee;"><strong>' . intval($attempts) . '</strong></td>
                </tr>
                <tr>
                    <td style="padding:10px;color:#666;">Status:</td>
                    <td style="padding:10px;"><span style="background-color:#dc3545;color:#fff;padding:3px 8px;border-radius:4px;font-size:12px;">IP blockiert</span></td>
                </tr>
            </table>
            <p>Die IP-Adresse wurde automatisch gesperrt.</p>
        ';
        $this->send_email($subject, $message, 'brute_force');
    }

    /**
     * Benachrichtigung: Malware gefunden
     */
    public function notify_malware($file, $threat_name) {
        $subject = 'Malware gefunden!';
        $message = '
            <p style="color:#dc3545;font-weight:600;">🚨 Verdächtige Datei auf Ihrer Website gefunden!</p>
            <table style="width:100%;border-collapse:collapse;margin:20px 0;">
                <tr>
                    <td style="padding:10px;border-bottom:1px solid #eee;color:#666;">Datei:</td>
                    <td style="padding:10px;border-bottom:1px solid #eee;"><code style="background:#f1f1f1;padding:2px 6px;border-radius:3px;">' . esc_html($file) . '</code></td>
                </tr>
                <tr>
                    <td style="padding:10px;color:#666;">Bedrohung:</td>
                    <td style="padding:10px;"><strong style="color:#dc3545;">' . esc_html($threat_name) . '</strong></td>
                </tr>
            </table>
            <p><strong>Empfohlene Maßnahmen:</strong></p>
            <ol>
                <li>Überprüfen Sie die Datei manuell</li>
                <li>Erstellen Sie ein Backup</li>
                <li>Löschen oder ersetzen Sie die infizierte Datei</li>
                <li>Ändern Sie alle Passwörter</li>
            </ol>
        ';
        $this->send_email($subject, $message, 'malware');
    }

    /**
     * Benachrichtigung: Neuer Admin-Login
     */
    public function notify_new_admin_login($user, $ip, $user_agent) {
        $subject = 'Neuer Administrator-Login';
        $location = $this->get_ip_location($ip);
        
        $message = '
            <p>👤 Ein Administrator hat sich eingeloggt:</p>
            <table style="width:100%;border-collapse:collapse;margin:20px 0;">
                <tr>
                    <td style="padding:10px;border-bottom:1px solid #eee;color:#666;">Benutzer:</td>
                    <td style="padding:10px;border-bottom:1px solid #eee;"><strong>' . esc_html($user->display_name) . '</strong> (' . esc_html($user->user_login) . ')</td>
                </tr>
                <tr>
                    <td style="padding:10px;border-bottom:1px solid #eee;color:#666;">IP-Adresse:</td>
                    <td style="padding:10px;border-bottom:1px solid #eee;"><strong>' . esc_html($ip) . '</strong></td>
                </tr>
                ' . ($location ? '<tr>
                    <td style="padding:10px;border-bottom:1px solid #eee;color:#666;">Standort:</td>
                    <td style="padding:10px;border-bottom:1px solid #eee;">' . esc_html($location) . '</td>
                </tr>' : '') . '
                <tr>
                    <td style="padding:10px;color:#666;">Browser:</td>
                    <td style="padding:10px;font-size:12px;">' . esc_html(substr($user_agent, 0, 100)) . '</td>
                </tr>
            </table>
            <p style="color:#666;">Wenn Sie diesen Login nicht durchgeführt haben, ändern Sie sofort Ihr Passwort!</p>
        ';
        $this->send_email($subject, $message, 'new_login');
    }

    /**
     * Benachrichtigung: Lizenz läuft ab
     */
    public function notify_license_expiring($days_remaining, $expiry_date) {
        $subject = 'SafetyPress Lizenz läuft bald ab';
        $color = $days_remaining <= 3 ? '#dc3545' : '#ffc107';
        
        $message = '
            <p>Ihre SafetyPress Pro Lizenz läuft in <strong style="color:' . $color . ';">' . intval($days_remaining) . ' Tagen</strong> ab.</p>
            <table style="width:100%;border-collapse:collapse;margin:20px 0;">
                <tr>
                    <td style="padding:10px;border-bottom:1px solid #eee;color:#666;">Ablaufdatum:</td>
                    <td style="padding:10px;border-bottom:1px solid #eee;"><strong>' . esc_html($expiry_date) . '</strong></td>
                </tr>
            </table>
            <p>Nach Ablauf werden Pro-Features deaktiviert.</p>
            <p style="text-align:center;margin:25px 0;">
                <a href="https://safetypress.de/kaufen" style="display:inline-block;background:#28a745;color:#fff;text-decoration:none;padding:12px 30px;border-radius:6px;font-weight:600;">Jetzt verlängern</a>
            </p>
        ';
        $this->send_email($subject, $message, 'license');
    }

    /**
     * Benachrichtigung: Core-Integrität (INFO - keine Malware!)
     */
    public function notify_integrity_issues($results) {
        $modified = $results['modified_files'] ?? [];
        $count = count($modified);
        
        if ($count === 0) return;
        
        $subject = 'Core-Dateien geändert (' . $count . ' Datei' . ($count > 1 ? 'en' : '') . ')';
        
        // Liste der geänderten Dateien
        $files_html = '<table style="width:100%;border-collapse:collapse;margin:20px 0;">';
        foreach (array_slice($modified, 0, 10) as $file) {
            $files_html .= '<tr>
                <td style="padding:8px;border-bottom:1px solid #eee;">📄 ' . esc_html($file['file']) . '</td>
                <td style="padding:8px;border-bottom:1px solid #eee;color:#666;font-size:12px;">Geändert: ' . esc_html($file['modified']) . '</td>
            </tr>';
        }
        $files_html .= '</table>';
        if ($count > 10) {
            $files_html .= '<p style="color:#666;">... und ' . ($count - 10) . ' weitere</p>';
        }
        
        $message = '
            <p style="color:#17a2b8;font-weight:600;">ℹ️ Core-Dateien weichen von der Original-Version ab</p>
            <p style="color:#666;font-size:14px;background:#f8f9fa;padding:12px;border-radius:6px;border-left:4px solid #17a2b8;">
                <strong>Hinweis:</strong> Dies ist <strong>keine Malware-Warnung</strong>. WordPress Core-Dateien können durch Updates, 
                Sprachpakete, Host-Optimierungen oder manuelle Anpassungen variieren. Prüfen Sie die Dateien bei Bedarf manuell.
            </p>
            ' . $files_html . '
            <p><strong>Empfohlene Maßnahmen:</strong></p>
            <ul style="color:#4a4a4a;">
                <li>Prüfen Sie ob kürzlich ein WordPress-Update durchgeführt wurde</li>
                <li>Bei Unsicherheit: Nutzen Sie die Auto-Repair Funktion im SafetyPress Dashboard</li>
                <li>Bei häufigen Änderungen: Kontaktieren Sie Ihren Hoster</li>
            </ul>
        ';
        $this->send_email($subject, $message, 'integrity');
    }

    /**
     * Benachrichtigung: Scan-Ergebnisse
     */
    public function notify_scan_results($results) {
        $issues = $results['issues'] ?? [];
        $stats = $results['stats'] ?? [];
        $scan_type = $results['scan_type'] ?? 'scheduled';
        
        $critical = 0;
        $warning = 0;
        $integrity = 0;
        
        foreach ($issues as $issue) {
            if ($issue['type'] === 'critical') $critical++;
            elseif ($issue['type'] === 'warning') $warning++;
            elseif ($issue['type'] === 'integrity') $integrity++;
        }
        
        $total = count($issues);
        $status_color = $critical > 0 ? '#dc3545' : ($warning > 0 ? '#ffc107' : '#28a745');
        $status_text = $critical > 0 ? 'Kritisch' : ($warning > 0 ? 'Warnung' : 'Alles OK');
        
        $subject = 'Scan-Ergebnis: ' . $total . ' Problem(e) gefunden';
        
        // Issue-Liste
        $issues_html = '';
        if (!empty($issues)) {
            $issues_html = '<table style="width:100%;border-collapse:collapse;margin:20px 0;">';
            foreach (array_slice($issues, 0, 15) as $issue) {
                $icon = $issue['type'] === 'critical' ? '🚨' : ($issue['type'] === 'warning' ? '⚠️' : ($issue['type'] === 'integrity' ? '🔧' : 'ℹ️'));
                $issues_html .= '<tr>
                    <td style="padding:8px;border-bottom:1px solid #eee;width:30px;">' . $icon . '</td>
                    <td style="padding:8px;border-bottom:1px solid #eee;">
                        <strong>' . esc_html($issue['message']) . '</strong><br>
                        <span style="color:#999;font-size:12px;">' . esc_html($issue['file'] ?? '') . '</span>
                    </td>
                </tr>';
            }
            $issues_html .= '</table>';
            if (count($issues) > 15) {
                $issues_html .= '<p style="color:#666;">... und ' . (count($issues) - 15) . ' weitere</p>';
            }
        }
        
        $message = '
            <p>Der geplante Sicherheitsscan wurde abgeschlossen:</p>
            <table style="width:100%;border-collapse:collapse;margin:20px 0;background:#f8f9fa;border-radius:6px;">
                <tr>
                    <td style="padding:20px;text-align:center;border-right:1px solid #eee;">
                        <div style="font-size:32px;font-weight:bold;color:' . $status_color . ';">' . $total . '</div>
                        <div style="color:#666;font-size:12px;">Probleme</div>
                    </td>
                    <td style="padding:20px;text-align:center;border-right:1px solid #eee;">
                        <div style="font-size:32px;font-weight:bold;">' . ($stats['files_scanned'] ?? 0) . '</div>
                        <div style="color:#666;font-size:12px;">Dateien gescannt</div>
                    </td>
                    <td style="padding:20px;text-align:center;">
                        <div style="font-size:24px;font-weight:bold;color:' . $status_color . ';">' . $status_text . '</div>
                        <div style="color:#666;font-size:12px;">Status</div>
                    </td>
                </tr>
            </table>
            ' . ($critical > 0 ? '<p style="background:#dc3545;color:#fff;padding:10px;border-radius:6px;">🚨 <strong>' . $critical . ' kritische Probleme</strong> erfordern sofortige Aufmerksamkeit!</p>' : '') . '
            ' . ($integrity > 0 ? '<p style="background:#17a2b8;color:#fff;padding:10px;border-radius:6px;">🔧 <strong>' . $integrity . ' Core-Dateien modifiziert</strong> - Nutzen Sie die Integritätsprüfung zur Reparatur.</p>' : '') . '
            ' . $issues_html . '
        ';
        
        $this->send_email($subject, $message, 'scan_results');
    }

    /**
     * Admin-Login tracken
     */
    public function track_admin_login($user_login, $user) {
        if (!user_can($user, 'manage_options')) {
            return;
        }
        
        $ip = $this->get_client_ip();
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field($_SERVER['HTTP_USER_AGENT']) : '';
        
        $known_ips = get_user_meta($user->ID, 'safetypress_known_ips', true) ?: [];
        
        if (!in_array($ip, $known_ips)) {
            do_action('safetypress_new_admin_login', $user, $ip, $user_agent);
            $known_ips[] = $ip;
            $known_ips = array_slice(array_unique($known_ips), -10);
            update_user_meta($user->ID, 'safetypress_known_ips', $known_ips);
        }
    }

    private function get_client_ip() {
        $keys = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
        foreach ($keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = explode(',', $_SERVER[$key])[0];
                if (filter_var(trim($ip), FILTER_VALIDATE_IP)) {
                    return trim($ip);
                }
            }
        }
        return '0.0.0.0';
    }

    private function get_ip_location($ip) {
        $cache_key = 'sp_geo_' . md5($ip);
        $cached = get_transient($cache_key);
        if ($cached !== false) return $cached;
        
        $response = wp_remote_get("http://ip-api.com/json/{$ip}?fields=country,city", ['timeout' => 3]);
        if (is_wp_error($response)) return '';
        
        $data = json_decode(wp_remote_retrieve_body($response));
        if (!$data || empty($data->country)) return '';
        
        $location = $data->city ? "{$data->city}, {$data->country}" : $data->country;
        set_transient($cache_key, $location, WEEK_IN_SECONDS);
        return $location;
    }

    /**
     * Test-E-Mail senden
     */
    public function send_test_email() {
        $message = '
            <p style="font-size:16px;">✅ Dies ist eine Test-Benachrichtigung von SafetyPress.</p>
            <p>Wenn Sie diese E-Mail erhalten haben, funktionieren die Benachrichtigungen korrekt.</p>
            <div style="text-align:center;padding:30px;background:#d4edda;border-radius:6px;margin:20px 0;">
                <div style="font-size:48px;">🎉</div>
                <div style="color:#155724;font-weight:bold;">Konfiguration erfolgreich!</div>
            </div>
        ';
        return $this->send_email('Test-Benachrichtigung', $message, 'test');
    }
}
