<?php
if (!defined('ABSPATH')) exit;

class SafetyPress_Login_Protection {

    private $attempts_table;

    public function __construct() {
        global $wpdb;
        $this->attempts_table = $wpdb->prefix . 'safetypress_login_attempts';
        
        if (get_option('safetypress_enable_login_protection', 1)) {
            add_action('wp_login_failed', array($this, 'handle_failed_login'));
            add_action('wp_login', array($this, 'handle_successful_login'), 10, 2);
            add_filter('authenticate', array($this, 'check_lockout'), 30, 3);
        }
    }

    public function handle_failed_login($username) {
        $ip = SafetyPress_Helper::get_client_ip();
        $this->log_attempt($ip, $username, false);
        
        // Activity Log: Fehlgeschlagener Login
        if (function_exists('safetypress') && safetypress()->logger) {
            safetypress()->logger->log('login_failed', sprintf('Fehlgeschlagener Login-Versuch für Benutzer: %s', $username), array('username' => $username));
        }
        
        $attempts = $this->get_recent_attempts($ip);
        $max = get_option('safetypress_max_login_attempts', 5);
        if ($attempts >= $max) {
            $this->lockout($ip);
            
            // Activity Log: IP gesperrt
            if (function_exists('safetypress') && safetypress()->logger) {
                safetypress()->logger->log('ip_locked', sprintf('IP %s wurde nach %d fehlgeschlagenen Versuchen gesperrt', $ip, $max), array('attempts' => $max));
            }
        }
    }

    public function handle_successful_login($user_login, $user) {
        $ip = SafetyPress_Helper::get_client_ip();
        $this->log_attempt($ip, $user_login, true);
        delete_transient('safetypress_lockout_' . md5($ip));
        
        // Activity Log: Erfolgreicher Login
        if (function_exists('safetypress') && safetypress()->logger) {
            safetypress()->logger->log('login_success', sprintf('Erfolgreicher Login: %s', $user_login), array('user_id' => $user->ID));
        }
    }

    public function check_lockout($user, $username, $password) {
        if (empty($username) || empty($password)) return $user;
        
        $ip = SafetyPress_Helper::get_client_ip();
        if (SafetyPress_Helper::is_whitelisted($ip)) return $user;
        
        $lockout = get_transient('safetypress_lockout_' . md5($ip));
        if ($lockout && $lockout > time()) {
            $remaining = ceil(($lockout - time()) / 60);
            
            // Activity Log: Blockierter Login-Versuch
            if (function_exists('safetypress') && safetypress()->logger) {
                safetypress()->logger->log('login_blocked', sprintf('Gesperrte IP %s versuchte Login als: %s', $ip, $username));
            }
            
            return new WP_Error('safetypress_lockout', 
                sprintf('<strong>FEHLER</strong>: Zu viele Versuche. Bitte warten Sie %d Minuten.', $remaining));
        }
        return $user;
    }

    private function log_attempt($ip, $username, $success) {
        global $wpdb;
        $wpdb->insert($this->attempts_table, array(
            'ip_address' => $ip,
            'username' => sanitize_user($username),
            'attempt_time' => current_time('mysql'),
            'success' => $success ? 1 : 0,
        ));
    }

    private function get_recent_attempts($ip) {
        global $wpdb;
        $duration = get_option('safetypress_lockout_duration', 30);
        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->attempts_table} WHERE ip_address = %s AND success = 0 AND attempt_time > DATE_SUB(NOW(), INTERVAL %d MINUTE)",
            $ip, $duration
        ));
    }

    private function lockout($ip) {
        $duration = get_option('safetypress_lockout_duration', 30);
        set_transient('safetypress_lockout_' . md5($ip), time() + ($duration * 60), $duration * 60);
    }

    public function get_statistics($days = 30) {
        global $wpdb;
        return array(
            'total_attempts' => $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->attempts_table} WHERE attempt_time > DATE_SUB(NOW(), INTERVAL %d DAY)", $days
            )),
            'failed_attempts' => $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->attempts_table} WHERE success = 0 AND attempt_time > DATE_SUB(NOW(), INTERVAL %d DAY)", $days
            )),
            'successful_logins' => $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->attempts_table} WHERE success = 1 AND attempt_time > DATE_SUB(NOW(), INTERVAL %d DAY)", $days
            )),
            'unique_ips' => $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(DISTINCT ip_address) FROM {$this->attempts_table} WHERE success = 0 AND attempt_time > DATE_SUB(NOW(), INTERVAL %d DAY)", $days
            )),
        );
    }
}
