<?php
/**
 * SafetyPress Login-Seite Anpassen
 * 
 * Installiert automatisch ein MU-Plugin für das URL-Routing.
 */

if (!defined('ABSPATH')) exit;

class SafetyPress_Login_Customizer {

    private $custom_slug;
    private $is_enabled;
    private $mu_plugin_path;

    public function __construct() {
        $this->custom_slug = get_option('safetypress_custom_login_url', '');
        $this->is_enabled = !empty($this->custom_slug) && get_option('safetypress_enable_custom_login', 0);
        $this->mu_plugin_path = ABSPATH . 'wp-content/mu-plugins/safetypress-login.php';
        
        // MU-Plugin bei Bedarf installieren/aktualisieren
        if ($this->is_enabled) {
            $this->ensure_mu_plugin();
        }
        
        // AJAX für Slug-Validierung
        add_action('wp_ajax_safetypress_validate_login_slug', [$this, 'ajax_validate_slug']);
    }

    /**
     * Stellt sicher dass das MU-Plugin existiert und aktuell ist
     */
    private function ensure_mu_plugin() {
        $mu_dir = ABSPATH . 'wp-content/mu-plugins';
        
        // Ordner erstellen falls nicht vorhanden
        if (!is_dir($mu_dir)) {
            wp_mkdir_p($mu_dir);
        }
        
        // MU-Plugin erstellen/aktualisieren
        $current_version = '1.1.0';
        $installed_version = get_option('safetypress_mu_plugin_version', '');
        
        if (!file_exists($this->mu_plugin_path) || $installed_version !== $current_version) {
            $this->install_mu_plugin();
            update_option('safetypress_mu_plugin_version', $current_version);
        }
    }

    /**
     * Installiert das MU-Plugin
     */
    private function install_mu_plugin() {
        $mu_plugin_code = $this->get_mu_plugin_code();
        file_put_contents($this->mu_plugin_path, $mu_plugin_code);
    }

    /**
     * Entfernt das MU-Plugin
     */
    public static function remove_mu_plugin() {
        $path = ABSPATH . 'wp-content/mu-plugins/safetypress-login.php';
        if (file_exists($path)) {
            unlink($path);
        }
        delete_option('safetypress_mu_plugin_version');
    }

    /**
     * MU-Plugin Code
     */
    private function get_mu_plugin_code() {
        return '<?php
/**
 * SafetyPress Login URL Handler
 * Automatisch installiert von SafetyPress
 * NICHT MANUELL BEARBEITEN!
 */

$request_uri = $_SERVER[\'REQUEST_URI\'] ?? \'\';

// Debug
if (strpos($request_uri, \'sp-debug\') !== false) {
    global $wpdb;
    header(\'Content-Type: text/plain; charset=utf-8\');
    echo "SafetyPress MU-Plugin aktiv!\n";
    if ($wpdb) {
        $slug = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'safetypress_custom_login_url\'");
        $enabled = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'safetypress_enable_custom_login\'");
        $redirect = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'safetypress_login_redirect_url\'");
        $home = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'home\'");
        echo "Slug: $slug\nEnabled: $enabled\nRedirect: $redirect\nHome: $home\n";
    }
    exit;
}

global $wpdb;
if (!$wpdb) return;

$custom_slug = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'safetypress_custom_login_url\'");
$enabled = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'safetypress_enable_custom_login\'");

if (empty($custom_slug) || !$enabled) return;

$home_url = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'home\'");
$redirect_url = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = \'safetypress_login_redirect_url\'");
$home_path = parse_url($home_url, PHP_URL_PATH);
$home_path = $home_path ? trim($home_path, \'/\') : \'\';

$path = parse_url($request_uri, PHP_URL_PATH);
$path = trim($path, \'/\');

$slug = $path;
if ($home_path && strpos($path, $home_path) === 0) {
    $slug = substr($path, strlen($home_path));
    $slug = ltrim($slug, \'/\');
}

// Custom Login URL -> Cookie setzen + Redirect zu wp-login.php
if ($slug === $custom_slug) {
    setcookie(\'sp_access\', md5($custom_slug), time() + 3600, \'/\');
    header(\'Location: \' . $home_url . \'/wp-login.php\');
    exit;
}

// Funktion fuer Blockierung (404 oder Redirect)
function sp_block_access($redirect_url, $home_url) {
    if (!empty($redirect_url)) {
        // Custom Redirect
        header(\'Location: \' . $redirect_url, true, 302);
        exit;
    } else {
        // Standard 404
        status_header(404);
        exit(\'404 - Not Found\');
    }
}

// wp-login.php blockieren ohne Cookie
if (strpos($request_uri, \'wp-login.php\') !== false) {
    if (isset($_COOKIE[\'sp_access\']) && $_COOKIE[\'sp_access\'] === md5($custom_slug)) {
        return;
    }
    sp_block_access($redirect_url, $home_url);
}

// wp-admin blockieren fuer nicht eingeloggte User
if (preg_match(\'#/wp-admin/?($|\?)#\', $request_uri)) {
    if (strpos($request_uri, \'admin-ajax.php\') !== false) return;
    if (strpos($request_uri, \'admin-post.php\') !== false) return;
    
    $logged_in = false;
    foreach ($_COOKIE as $name => $value) {
        if (strpos($name, \'wordpress_logged_in_\') === 0) {
            $logged_in = true;
            break;
        }
    }
    
    if (!$logged_in) {
        sp_block_access($redirect_url, $home_url);
    }
}
';
    }

    /**
     * AJAX: Slug validieren
     */
    public function ajax_validate_slug() {
        check_ajax_referer('safetypress_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Keine Berechtigung']);
        }
        
        $slug = isset($_POST['slug']) ? sanitize_title($_POST['slug']) : '';
        
        if (empty($slug)) {
            wp_send_json_error(['message' => 'Slug darf nicht leer sein']);
        }
        
        if (strlen($slug) < 4) {
            wp_send_json_error(['message' => 'Slug muss mindestens 4 Zeichen lang sein']);
        }
        
        $reserved = ['wp-admin', 'wp-login', 'wp-content', 'wp-includes', 'admin', 'login', 'dashboard', 'feed', 'rss', 'sitemap'];
        if (in_array($slug, $reserved)) {
            wp_send_json_error(['message' => 'Dieser Slug ist reserviert']);
        }
        
        $page = get_page_by_path($slug);
        if ($page) {
            wp_send_json_error(['message' => 'Dieser Slug wird bereits von einer Seite verwendet']);
        }
        
        wp_send_json_success([
            'message' => 'Slug ist verfügbar',
            'slug' => $slug,
            'preview_url' => home_url($slug),
        ]);
    }

    /**
     * Status abrufen
     */
    public function get_status() {
        return [
            'enabled' => $this->is_enabled,
            'custom_slug' => $this->custom_slug,
            'custom_url' => $this->custom_slug ? home_url($this->custom_slug) : '',
            'default_url' => home_url('wp-login.php'),
            'mu_plugin_installed' => file_exists($this->mu_plugin_path),
        ];
    }

    /**
     * Speichert die Einstellungen
     */
    public static function save_settings($slug, $enabled) {
        $slug = sanitize_title($slug);
        
        update_option('safetypress_custom_login_url', $slug);
        update_option('safetypress_enable_custom_login', $enabled ? 1 : 0);
        
        // MU-Plugin installieren oder entfernen
        if ($enabled && !empty($slug)) {
            $instance = new self();
            $instance->ensure_mu_plugin();
        } else {
            self::remove_mu_plugin();
        }
        
        return true;
    }
}
