<?php
if (!defined('ABSPATH')) exit;

class SafetyPress_Logger {

    private $table_name;

    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'safetypress_logs';
        add_action('safetypress_cleanup', array($this, 'cleanup'));
    }

    public function log($event_type, $message, $extra_data = array()) {
        global $wpdb;
        if (!get_option('safetypress_enable_activity_log', 1)) return false;
        
        $wpdb->insert($this->table_name, array(
            'event_type' => sanitize_key($event_type),
            'event_message' => sanitize_text_field($message),
            'ip_address' => SafetyPress_Helper::get_client_ip(),
            'user_id' => get_current_user_id(),
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field($_SERVER['HTTP_USER_AGENT']) : '',
            'extra_data' => !empty($extra_data) ? wp_json_encode($extra_data) : null,
            'created_at' => current_time('mysql'),
        ));
        
        do_action('safetypress_event_logged', $event_type, $message, $extra_data);
        return true;
    }

    public function get_logs($args = array()) {
        global $wpdb;
        $defaults = array('limit' => 100, 'offset' => 0);
        $args = wp_parse_args($args, $defaults);
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->table_name} ORDER BY created_at DESC LIMIT %d OFFSET %d",
            absint($args['limit']), absint($args['offset'])
        ));
    }

    public function get_count($event_type = '') {
        global $wpdb;
        if (!empty($event_type)) {
            return $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->table_name} WHERE event_type = %s", $event_type
            ));
        }
        return $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name}");
    }

    public function export_csv($logs) {
        $csv = "ID,Event,Message,IP,User ID,Date\n";
        foreach ($logs as $log) {
            $csv .= sprintf("%d,%s,%s,%s,%d,%s\n",
                $log->id, $log->event_type, 
                '"' . str_replace('"', '""', $log->event_message) . '"',
                $log->ip_address, $log->user_id, $log->created_at
            );
        }
        return $csv;
    }

    public function cleanup() {
        global $wpdb;
        $days = get_option('safetypress_log_retention_days', 30);
        $wpdb->query($wpdb->prepare(
            "DELETE FROM {$this->table_name} WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)", absint($days)
        ));
    }
}
