<?php
/**
 * SafetyPress License Management
 * 
 * Prüft Lizenzstatus über die SafetyPress API.
 */

if (!defined('ABSPATH')) exit;

class SafetyPress_License {

    private $license_key;

    public function __construct() {
        $this->license_key = get_option('safetypress_license_key', '');
        
        add_action('wp_ajax_safetypress_activate_license', array($this, 'ajax_activate'));
        add_action('wp_ajax_safetypress_deactivate_license', array($this, 'ajax_deactivate'));
    }

    /**
     * Prüft ob Pro-Lizenz aktiv ist
     */
    public function is_valid() {
        // Dev-Mode via wp-config.php
        if (defined('SAFETYPRESS_PRO') && SAFETYPRESS_PRO === true) {
            return true;
        }
        
        // Status aus Datenbank prüfen
        $status = get_option('safetypress_license_status', 'free');
        return in_array($status, array('valid', 'active', 'lifetime', 'trialing'));
    }

    /**
     * Ist dies eine Stripe-Subscription?
     */
    public function is_stripe_subscription() {
        return get_option('safetypress_license_type', '') === 'subscription';
    }

    /**
     * Ist AppSumo-Lizenz?
     */
    public function is_appsumo() {
        $type = get_option('safetypress_license_type', '');
        return $type === 'appsumo';
    }

    /**
     * Ist Lifetime-Lizenz?
     */
    public function is_lifetime() {
        $type = get_option('safetypress_license_type', '');
        return in_array($type, array('lifetime', 'appsumo', 'developer'));
    }

    /**
     * Hole Tier
     */
    public function get_tier() {
        if (!$this->is_valid()) return 0;
        return (int) get_option('safetypress_license_tier', 1);
    }

    /**
     * Erlaubte Sites
     */
    public function get_sites_allowed() {
        $tiers = array(0 => 0, 1 => 1, 2 => 3, 3 => 10, 4 => 999);
        $tier = $this->get_tier();
        return isset($tiers[$tier]) ? $tiers[$tier] : 1;
    }

    /**
     * Hat Feature?
     */
    public function has_feature($feature) {
        if (!$this->is_valid()) return false;
        $tier = $this->get_tier();
        $features = array(
            'two_factor' => 1, 
            'advanced_firewall' => 1, 
            'malware_scanner' => 1,
            'priority_support' => 2, 
            'white_label' => 3, 
            'agency_dashboard' => 4
        );
        return $tier >= (isset($features[$feature]) ? $features[$feature] : 1);
    }

    /**
     * Lizenz-Info für Templates
     */
    public function get_license_info() {
        $key = $this->license_key;
        
        return array(
            'key' => strlen($key) > 8 ? substr($key, 0, 4) . '****' . substr($key, -4) : $key,
            'status' => get_option('safetypress_license_status', 'free'),
            'type' => get_option('safetypress_license_type', ''),
            'tier' => $this->get_tier(),
            'is_valid' => $this->is_valid(),
            'is_appsumo' => $this->is_appsumo(),
            'is_lifetime' => $this->is_lifetime(),
            'is_stripe' => $this->is_stripe_subscription(),
            'sites_allowed' => $this->get_sites_allowed(),
            'activated' => get_option('safetypress_license_activated', ''),
            'expires' => get_option('safetypress_license_expires', ''),
            'warning' => get_option('safetypress_license_warning', ''),
            'stripe_status' => get_option('safetypress_stripe_subscription_status', ''),
        );
    }

    /**
     * AJAX: Lizenz aktivieren
     */
    public function ajax_activate() {
        check_ajax_referer('safetypress_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        }
        
        $key = isset($_POST['license_key']) ? sanitize_text_field($_POST['license_key']) : '';
        
        if (empty($key)) {
            wp_send_json_error(array('message' => 'Bitte Lizenzschlüssel eingeben.'));
        }
        
        // Aktivierung über API
        if (function_exists('safetypress') && safetypress()->stripe) {
            $result = safetypress()->stripe->activate_license($key);
            
            if (isset($result['success']) && $result['success']) {
                wp_send_json_success($result);
            } else {
                wp_send_json_error(array(
                    'message' => isset($result['message']) ? $result['message'] : 'Aktivierung fehlgeschlagen.'
                ));
            }
        } else {
            wp_send_json_error(array('message' => 'Stripe-Modul nicht verfügbar.'));
        }
    }

    /**
     * AJAX: Lizenz deaktivieren
     */
    public function ajax_deactivate() {
        check_ajax_referer('safetypress_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        }
        
        // Deaktivierung über API
        if (function_exists('safetypress') && safetypress()->stripe) {
            $result = safetypress()->stripe->deactivate_license();
        }
        
        // Lokale Daten immer löschen
        delete_option('safetypress_license_key');
        update_option('safetypress_license_status', 'free');
        delete_option('safetypress_license_type');
        delete_option('safetypress_license_tier');
        delete_option('safetypress_stripe_subscription_status');
        delete_option('safetypress_license_warning');
        delete_option('safetypress_license_expires');
        delete_option('safetypress_license_activated');
        delete_transient('safetypress_license_cache');
        
        $this->license_key = '';
        
        wp_send_json_success(array('message' => 'Lizenz deaktiviert.'));
    }
}
