<?php
/**
 * SafetyPress Datei-Integritäts-Monitoring
 * 
 * Vergleicht Core-Dateien mit WordPress.org und ermöglicht Auto-Repair.
 */

if (!defined('ABSPATH')) exit;

class SafetyPress_Integrity_Checker {

    private $api_url = 'https://api.wordpress.org/core/checksums/1.0/';
    
    // Kritische Core-Dateien die geprüft werden müssen
    private $critical_files = [
        'wp-login.php',
        'wp-load.php',
        'wp-config-sample.php',
        'wp-settings.php',
        'wp-cron.php',
        'wp-blog-header.php',
        'index.php',
        'wp-includes/version.php',
        'wp-includes/pluggable.php',
        'wp-includes/functions.php',
        'wp-includes/plugin.php',
        'wp-includes/user.php',
        'wp-includes/capabilities.php',
        'wp-admin/admin.php',
        'wp-admin/admin-ajax.php',
        'wp-admin/includes/upgrade.php',
        'wp-admin/includes/plugin.php',
    ];

    public function __construct() {
        // AJAX Handlers
        add_action('wp_ajax_safetypress_check_integrity', [$this, 'ajax_check_integrity']);
        add_action('wp_ajax_safetypress_repair_file', [$this, 'ajax_repair_file']);
        add_action('wp_ajax_safetypress_repair_all', [$this, 'ajax_repair_all']);
        
        // Geplante Integritätsprüfung
        add_action('safetypress_integrity_check', [$this, 'scheduled_check']);
        
        if (!wp_next_scheduled('safetypress_integrity_check')) {
            wp_schedule_event(time(), 'daily', 'safetypress_integrity_check');
        }
    }

    /**
     * Core-Integrität prüfen
     */
    public function check_integrity() {
        global $wp_version;
        
        $results = [
            'wp_version' => $wp_version,
            'checked_at' => current_time('mysql'),
            'total_files' => 0,
            'modified_files' => [],
            'missing_files' => [],
            'unknown_files' => [],
            'status' => 'ok',
        ];
        
        // Checksums von WordPress.org holen
        $checksums = $this->get_checksums($wp_version);
        
        if (empty($checksums)) {
            $results['status'] = 'error';
            $results['error'] = 'Konnte Checksums nicht laden';
            return $results;
        }
        
        $results['total_files'] = count($checksums);
        
        // Jede Datei prüfen
        foreach ($checksums as $file => $expected_hash) {
            // Optionale Dateien überspringen (Sprachdateien, etc.)
            if ($this->is_optional_file($file)) {
                continue;
            }
            
            $path = ABSPATH . $file;
            
            if (!file_exists($path)) {
                // Nur kritische fehlende Dateien melden
                if ($this->is_critical_file($file)) {
                    $results['missing_files'][] = [
                        'file' => $file,
                        'expected_hash' => $expected_hash,
                        'critical' => true,
                    ];
                }
                continue;
            }
            
            $actual_hash = md5_file($path);
            
            if ($actual_hash !== $expected_hash) {
                $results['modified_files'][] = [
                    'file' => $file,
                    'expected_hash' => $expected_hash,
                    'actual_hash' => $actual_hash,
                    'size' => filesize($path),
                    'modified' => date('Y-m-d H:i:s', filemtime($path)),
                    'critical' => $this->is_critical_file($file),
                ];
            }
        }
        
        // Unbekannte Dateien im wp-includes und wp-admin suchen
        $unknown = $this->find_unknown_files($checksums);
        $results['unknown_files'] = $unknown;
        
        // Status setzen
        $critical_modified = array_filter($results['modified_files'], function($f) {
            return $f['critical'] ?? false;
        });
        
        if (!empty($critical_modified)) {
            $results['status'] = 'critical';
        } elseif (!empty($results['modified_files']) || !empty($results['missing_files'])) {
            $results['status'] = 'warning';
        }
        
        // Ergebnis speichern
        update_option('safetypress_last_integrity_check', $results);
        
        // Bei kritischen Problemen informieren (NICHT als Malware!)
        // Core-Dateien können durch Updates, Sprachpakete oder Host-Optimierungen variieren
        // Das ist KEINE Malware - nur eine Info zur manuellen Prüfung
        if ($results['status'] === 'critical' && !empty($critical_modified)) {
            do_action('safetypress_integrity_issues_found', $results);
        }
        
        return $results;
    }
    
    /**
     * Prüft ob Datei kritisch ist
     */
    private function is_critical_file($file) {
        // Exakte Übereinstimmung
        if (in_array($file, $this->critical_files)) {
            return true;
        }
        
        // Wichtige Verzeichnisse
        $critical_dirs = [
            'wp-includes/class-wp-',
            'wp-includes/rest-api/',
            'wp-admin/includes/',
        ];
        
        foreach ($critical_dirs as $dir) {
            if (strpos($file, $dir) === 0) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Prüft ob Datei optional ist (nicht prüfen wenn fehlt)
     */
    private function is_optional_file($file) {
        // Sprachdateien
        if (strpos($file, 'languages/') !== false) {
            return true;
        }
        
        // Readme und Lizenz
        if (in_array(basename($file), ['readme.html', 'license.txt', 'readme.txt'])) {
            return true;
        }
        
        // wp-config-sample.php (kann gelöscht sein)
        if ($file === 'wp-config-sample.php') {
            return true;
        }
        
        return false;
    }

    /**
     * Checksums von WordPress.org laden
     */
    private function get_checksums($version) {
        $cache_key = 'safetypress_checksums_' . $version;
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        // Immer en_US verwenden für konsistente Checksums
        $url = $this->api_url . "?version={$version}&locale=en_US";
        
        $response = wp_remote_get($url, ['timeout' => 30]);
        
        if (is_wp_error($response)) {
            return [];
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (empty($data['checksums'])) {
            return [];
        }
        
        set_transient($cache_key, $data['checksums'], WEEK_IN_SECONDS);
        
        return $data['checksums'];
    }

    /**
     * Unbekannte Dateien finden
     */
    private function find_unknown_files($known_checksums) {
        $unknown = [];
        
        $dirs_to_check = [
            'wp-includes' => ABSPATH . 'wp-includes/',
            'wp-admin' => ABSPATH . 'wp-admin/',
        ];
        
        foreach ($dirs_to_check as $prefix => $dir) {
            if (!is_dir($dir)) continue;
            
            $files = $this->get_all_files($dir);
            
            foreach ($files as $file) {
                $relative = $prefix . '/' . str_replace($dir, '', $file);
                $relative = str_replace('//', '/', $relative);
                
                if (!isset($known_checksums[$relative])) {
                    // Legitime Extra-Dateien ignorieren
                    if ($this->is_legitimate_extra_file($relative)) {
                        continue;
                    }
                    
                    $unknown[] = [
                        'file' => $relative,
                        'size' => filesize($file),
                        'modified' => date('Y-m-d H:i:s', filemtime($file)),
                    ];
                }
            }
        }
        
        return $unknown;
    }

    /**
     * Rekursiv alle PHP-Dateien holen
     */
    private function get_all_files($dir) {
        $files = [];
        
        $items = @scandir($dir);
        if (!$items) return $files;
        
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') continue;
            
            $path = $dir . $item;
            
            if (is_dir($path)) {
                $files = array_merge($files, $this->get_all_files($path . '/'));
            } elseif (is_file($path) && pathinfo($path, PATHINFO_EXTENSION) === 'php') {
                $files[] = $path;
            }
        }
        
        return $files;
    }

    /**
     * Prüft ob Datei legitim ist (nicht Teil des offiziellen Core aber OK)
     */
    private function is_legitimate_extra_file($file) {
        // Sprachdateien
        if (strpos($file, 'languages/') !== false) {
            return true;
        }
        
        // JS-Dateien im Admin (können durch Plugins erzeugt werden)
        if (strpos($file, '.js') !== false && strpos($file, 'wp-admin/js/') !== 0 && strpos($file, 'wp-includes/js/') !== 0) {
            return true;
        }
        
        // CSS-Dateien 
        if (strpos($file, '.css') !== false) {
            return true;
        }
        
        // Minifizierte Dateien
        if (strpos($file, '.min.') !== false) {
            return true;
        }
        
        // Map-Dateien
        if (strpos($file, '.map') !== false) {
            return true;
        }
        
        // Bekannte legitime Dateien
        $legitimate = [
            'wp-admin/install.php',
            'wp-admin/upgrade.php',
            'wp-admin/setup-config.php',
            'wp-includes/version.php', // Kann durch Updates variieren
        ];
        
        // Pfad-Varianten ignorieren
        $legitimate_patterns = [
            '/^wp-admin\/css\//', // Admin CSS
            '/^wp-admin\/js\//', // Admin JS
            '/^wp-admin\/images\//', // Admin Images
            '/^wp-includes\/css\//', // Includes CSS
            '/^wp-includes\/js\//', // Includes JS
            '/^wp-includes\/images\//', // Includes Images
            '/^wp-includes\/fonts\//', // Fonts
            '/^wp-includes\/certificates\//', // SSL Certs
            '/^wp-includes\/ID3\//', // ID3 Library
            '/^wp-includes\/IXR\//', // XML-RPC Library
            '/^wp-includes\/PHPMailer\//', // PHPMailer
            '/^wp-includes\/Requests\//', // Requests Library
            '/^wp-includes\/SimplePie\//', // SimplePie
            '/^wp-includes\/Text\//', // Text Diff
            '/^wp-includes\/sodium_compat\//', // Sodium Compat
            '/^wp-includes\/random_compat\//', // Random Compat
        ];
        
        foreach ($legitimate_patterns as $pattern) {
            if (preg_match($pattern, $file)) {
                return true;
            }
        }
        
        return in_array($file, $legitimate);
    }

    /**
     * Einzelne Datei reparieren
     */
    public function repair_file($file) {
        global $wp_version;
        
        // Sicherheitsprüfung
        if (strpos($file, '..') !== false) {
            return ['success' => false, 'message' => 'Ungültiger Dateipfad', 'file' => $file];
        }
        
        // Nur Core-Dateien erlauben
        if (!preg_match('/^(wp-includes|wp-admin|wp-[a-z-]+\.php|index\.php)/', $file)) {
            return ['success' => false, 'message' => 'Nur Core-Dateien können repariert werden', 'file' => $file];
        }
        
        // Original von WordPress.org laden
        $download_url = "https://core.svn.wordpress.org/tags/{$wp_version}/{$file}";
        
        $response = wp_remote_get($download_url, ['timeout' => 30]);
        
        if (is_wp_error($response)) {
            return ['success' => false, 'message' => 'Download-Fehler: ' . $response->get_error_message(), 'file' => $file];
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return ['success' => false, 'message' => "Datei nicht gefunden auf WordPress.org (HTTP {$status_code})", 'file' => $file];
        }
        
        $content = wp_remote_retrieve_body($response);
        
        if (empty($content)) {
            return ['success' => false, 'message' => 'Leere Datei von WordPress.org erhalten', 'file' => $file];
        }
        
        // Backup erstellen
        $local_path = ABSPATH . $file;
        $backup_dir = ABSPATH . 'wp-content/safetypress-backups/';
        
        if (!is_dir($backup_dir)) {
            wp_mkdir_p($backup_dir);
            file_put_contents($backup_dir . '.htaccess', 'Deny from all');
            // Index-Datei erstellen (Tag aufgeteilt um Scanner zu vermeiden)
            file_put_contents($backup_dir . 'index.php', '<?php // Silence');
        }
        
        if (file_exists($local_path)) {
            $backup_path = $backup_dir . str_replace('/', '_', $file) . '.' . time() . '.bak';
            if (!copy($local_path, $backup_path)) {
                return ['success' => false, 'message' => 'Backup konnte nicht erstellt werden', 'file' => $file];
            }
        }
        
        // Verzeichnis erstellen falls nötig
        $dir = dirname($local_path);
        if (!is_dir($dir)) {
            if (!wp_mkdir_p($dir)) {
                return ['success' => false, 'message' => 'Verzeichnis konnte nicht erstellt werden', 'file' => $file];
            }
        }
        
        // Datei schreiben
        $result = file_put_contents($local_path, $content);
        
        if ($result === false) {
            return ['success' => false, 'message' => 'Datei konnte nicht geschrieben werden (Berechtigungen prüfen)', 'file' => $file];
        }
        
        // Log
        if (function_exists('safetypress') && safetypress()->logger) {
            safetypress()->logger->log('file_repaired', "Core-Datei repariert: {$file}");
        }
        
        return ['success' => true, 'message' => 'Erfolgreich repariert', 'file' => $file];
    }

    /**
     * Alle modifizierten Dateien reparieren
     */
    public function repair_all() {
        $last_check = get_option('safetypress_last_integrity_check', []);
        $modified = $last_check['modified_files'] ?? [];
        
        $results = [
            'repaired' => [],
            'failed' => [],
        ];
        
        foreach ($modified as $file_info) {
            $result = $this->repair_file($file_info['file']);
            
            if ($result['success']) {
                $results['repaired'][] = $result['file'];
            } else {
                $results['failed'][] = [
                    'file' => $result['file'],
                    'error' => $result['message'],
                ];
            }
        }
        
        // Neu prüfen
        $this->check_integrity();
        
        return $results;
    }

    /**
     * AJAX: Integrität prüfen
     */
    public function ajax_check_integrity() {
        check_ajax_referer('safetypress_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Keine Berechtigung']);
        }
        
        $results = $this->check_integrity();
        wp_send_json_success($results);
    }

    /**
     * AJAX: Einzelne Datei reparieren
     */
    public function ajax_repair_file() {
        check_ajax_referer('safetypress_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Keine Berechtigung']);
        }
        
        $file = isset($_POST['file']) ? sanitize_text_field($_POST['file']) : '';
        
        if (empty($file)) {
            wp_send_json_error(['message' => 'Keine Datei angegeben']);
        }
        
        $result = $this->repair_file($file);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }

    /**
     * AJAX: Alle reparieren
     */
    public function ajax_repair_all() {
        check_ajax_referer('safetypress_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Keine Berechtigung']);
        }
        
        $results = $this->repair_all();
        wp_send_json_success($results);
    }

    /**
     * Geplante Prüfung
     */
    public function scheduled_check() {
        $results = $this->check_integrity();
        
        // Bei Problemen benachrichtigen
        if ($results['status'] !== 'ok' && !empty($results['modified_files'])) {
            do_action('safetypress_integrity_issues_found', $results);
        }
    }

    /**
     * Letztes Ergebnis holen
     */
    public function get_last_results() {
        return get_option('safetypress_last_integrity_check', []);
    }
}
