<?php
if (!defined('ABSPATH')) exit;

class SafetyPress_Helper {

    public static function get_client_ip() {
        $ip = '';
        if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) {
            $ip = $_SERVER['HTTP_CF_CONNECTING_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
            $ip = trim($ips[0]);
        } elseif (!empty($_SERVER['HTTP_X_REAL_IP'])) {
            $ip = $_SERVER['HTTP_X_REAL_IP'];
        } elseif (!empty($_SERVER['REMOTE_ADDR'])) {
            $ip = $_SERVER['REMOTE_ADDR'];
        }
        return filter_var($ip, FILTER_VALIDATE_IP) ? $ip : '0.0.0.0';
    }

    public static function is_whitelisted($ip) {
        $whitelist = apply_filters('safetypress_ip_whitelist', array('127.0.0.1', '::1'));
        return in_array($ip, $whitelist);
    }

    public static function get_country_from_ip($ip) {
        $key = 'safetypress_geo_' . md5($ip);
        $country = get_transient($key);
        if ($country === false) {
            $response = wp_remote_get("http://ip-api.com/json/{$ip}?fields=countryCode", array('timeout' => 5));
            if (!is_wp_error($response)) {
                $data = json_decode(wp_remote_retrieve_body($response), true);
                $country = isset($data['countryCode']) ? $data['countryCode'] : 'XX';
            } else {
                $country = 'XX';
            }
            set_transient($key, $country, DAY_IN_SECONDS);
        }
        return $country;
    }

    public static function is_bad_bot() {
        $ua = isset($_SERVER['HTTP_USER_AGENT']) ? strtolower($_SERVER['HTTP_USER_AGENT']) : '';
        $bots = array('ahrefsbot', 'semrushbot', 'dotbot', 'mj12bot', 'blexbot');
        foreach ($bots as $bot) {
            if (strpos($ua, $bot) !== false) return true;
        }
        return false;
    }

    /**
     * Berechnet den Security Score
     * 
     * FREE User: Max 70% möglich (PRO-Features fehlen)
     * PRO User: Max 100% möglich
     * 
     * @return int Score von 0-100
     */
    public static function get_security_score() {
        $score = 0;
        $details = self::get_score_details();
        
        foreach ($details as $item) {
            if ($item['active']) {
                $score += $item['points'];
            }
        }
        
        return min($score, 100);
    }
    
    /**
     * Detaillierte Score-Aufschlüsselung
     * Wird für Dashboard-Anzeige verwendet
     * 
     * @return array
     */
    public static function get_score_details() {
        $is_pro = function_exists('safetypress') && safetypress()->is_pro();
        
        $items = array(
            // FREE Features (max 70 Punkte)
            array(
                'id' => 'firewall',
                'name' => 'Firewall',
                'description' => 'Enterprise Firewall aktiv',
                'points' => 15,
                'active' => (bool) get_option('safetypress_enable_firewall', 1),
                'pro_only' => false,
            ),
            array(
                'id' => 'login_protection',
                'name' => 'Login-Schutz',
                'description' => 'Brute-Force Prevention',
                'points' => 15,
                'active' => (bool) get_option('safetypress_enable_login_protection', 1),
                'pro_only' => false,
            ),
            array(
                'id' => 'hardening',
                'name' => 'WordPress Hardening',
                'description' => 'Security Headers & Härtung',
                'points' => 10,
                'active' => (bool) get_option('safetypress_enable_hardening', 1),
                'pro_only' => false,
            ),
            array(
                'id' => 'xmlrpc',
                'name' => 'XML-RPC deaktiviert',
                'description' => 'Angriffsfläche reduziert',
                'points' => 5,
                'active' => (bool) get_option('safetypress_disable_xmlrpc', 1),
                'pro_only' => false,
            ),
            array(
                'id' => 'file_editor',
                'name' => 'Datei-Editor deaktiviert',
                'description' => 'Kein Code-Editing im Admin',
                'points' => 5,
                'active' => (bool) get_option('safetypress_disable_file_editor', 1),
                'pro_only' => false,
            ),
            array(
                'id' => 'ssl',
                'name' => 'SSL/HTTPS',
                'description' => 'Verschlüsselte Verbindung',
                'points' => 10,
                'active' => is_ssl(),
                'pro_only' => false,
            ),
            array(
                'id' => 'activity_log',
                'name' => 'Aktivitäts-Log',
                'description' => 'Ereignisse werden protokolliert',
                'points' => 10,
                'active' => (bool) get_option('safetypress_enable_activity_log', 1),
                'pro_only' => false,
            ),
            
            // PRO Features (30 Punkte)
            array(
                'id' => '2fa',
                'name' => 'Zwei-Faktor-Auth',
                'description' => 'TOTP Authenticator',
                'points' => 10,
                'active' => $is_pro && (bool) get_option('safetypress_enable_2fa', 0),
                'pro_only' => true,
                'available' => $is_pro,
            ),
            array(
                'id' => 'malware_scanner',
                'name' => 'Malware Deep-Scan',
                'description' => 'Erweiterte Malware-Erkennung',
                'points' => 10,
                'active' => $is_pro, // Automatisch aktiv wenn PRO
                'pro_only' => true,
                'available' => $is_pro,
            ),
            array(
                'id' => 'woo_protection',
                'name' => 'WooCommerce Schutz',
                'description' => 'Betrugs- und Spam-Schutz',
                'points' => 10,
                'active' => $is_pro && class_exists('WooCommerce') && (bool) get_option('safetypress_enable_woo_protection', 1),
                'pro_only' => true,
                'available' => $is_pro && class_exists('WooCommerce'),
            ),
        );
        
        return $items;
    }
    
    /**
     * Gibt den maximalen erreichbaren Score zurück
     * FREE: 70, PRO: 100
     * 
     * @return int
     */
    public static function get_max_score() {
        $is_pro = function_exists('safetypress') && safetypress()->is_pro();
        return $is_pro ? 100 : 70;
    }
    
    /**
     * Gibt zurück wie viele Punkte durch PRO-Upgrade möglich wären
     * 
     * @return int
     */
    public static function get_potential_pro_points() {
        $is_pro = function_exists('safetypress') && safetypress()->is_pro();
        if ($is_pro) return 0;
        
        $pro_points = 0;
        $details = self::get_score_details();
        
        foreach ($details as $item) {
            if ($item['pro_only']) {
                $pro_points += $item['points'];
            }
        }
        
        return $pro_points;
    }

    public static function send_notification($subject, $message, $type = 'info') {
        if (!get_option('safetypress_email_notifications', 1)) return false;
        $to = get_option('safetypress_notification_email', get_option('admin_email'));
        $headers = array('Content-Type: text/html; charset=UTF-8');
        $html = '<div style="font-family:Arial,sans-serif;max-width:600px;margin:0 auto;">';
        $html .= '<div style="background:#2271b1;color:white;padding:20px;text-align:center;"><h1>SafetyPress</h1></div>';
        $html .= '<div style="padding:30px;background:#f5f5f5;">' . wp_kses_post($message) . '</div></div>';
        return wp_mail($to, '[SafetyPress] ' . $subject, $html, $headers);
    }

    public static function get_high_risk_countries() {
        return array('RU', 'CN', 'KP', 'IR');
    }
}
