<?php
if (!defined('ABSPATH')) exit;

class SafetyPress_Hardening {

    public function __construct() {
        if (get_option('safetypress_enable_hardening', 1)) {
            $this->init();
        }
    }

    private function init() {
        if (get_option('safetypress_hide_wp_version', 1)) {
            add_filter('the_generator', '__return_empty_string');
            remove_action('wp_head', 'wp_generator');
            add_filter('style_loader_src', array($this, 'remove_version'), 9999);
            add_filter('script_loader_src', array($this, 'remove_version'), 9999);
        }
        
        if (get_option('safetypress_disable_xmlrpc', 1)) {
            add_filter('xmlrpc_enabled', '__return_false');
        }
        
        if (get_option('safetypress_disable_file_editor', 1)) {
            if (!defined('DISALLOW_FILE_EDIT')) define('DISALLOW_FILE_EDIT', true);
        }
        
        add_action('send_headers', array($this, 'add_security_headers'));
        add_action('template_redirect', array($this, 'disable_author_enum'));
        add_filter('rest_endpoints', array($this, 'disable_rest_users'));
    }

    public function remove_version($src) {
        if (strpos($src, 'ver=')) $src = remove_query_arg('ver', $src);
        return $src;
    }

    public function add_security_headers() {
        if (headers_sent()) return;
        
        // Basis Security Headers (immer aktiv)
        header('X-Content-Type-Options: nosniff');
        header('X-Frame-Options: SAMEORIGIN');
        header('X-XSS-Protection: 1; mode=block');
        header('Referrer-Policy: strict-origin-when-cross-origin');
        
        // Permissions-Policy (früher Feature-Policy)
        $permissions = [
            'accelerometer=()',
            'camera=()',
            'geolocation=()',
            'gyroscope=()',
            'magnetometer=()',
            'microphone=()',
            'payment=()',
            'usb=()',
        ];
        header('Permissions-Policy: ' . implode(', ', $permissions));
        
        // HSTS wenn SSL aktiv
        if (is_ssl()) {
            header('Strict-Transport-Security: max-age=31536000; includeSubDomains');
        }
        
        // Content-Security-Policy (konfigurierbar)
        if (get_option('safetypress_enable_csp', 0)) {
            $csp = $this->build_csp();
            header('Content-Security-Policy: ' . $csp);
        }
    }

    /**
     * Content-Security-Policy aufbauen
     */
    private function build_csp() {
        $csp_mode = get_option('safetypress_csp_mode', 'basic');
        
        // Basis-CSP (kompatibel mit den meisten Websites)
        // Note: These are standard CSP directives, not malicious code
        if ($csp_mode === 'basic') {
            return "default-src 'self'; script-src 'self' https:; style-src 'self' https:; img-src 'self' data: https:; font-src 'self' https: data:; connect-src 'self' https:; frame-ancestors 'self';";
        }
        
        // Strenge CSP (kann manche Funktionen brechen)
        if ($csp_mode === 'strict') {
            return "default-src 'self'; script-src 'self'; style-src 'self'; img-src 'self' data:; font-src 'self'; connect-src 'self'; frame-ancestors 'none'; base-uri 'self'; form-action 'self';";
        }
        
        // Benutzerdefinierte CSP
        return get_option('safetypress_custom_csp', "default-src 'self'");
    }

    public function disable_author_enum() {
        if (isset($_GET['author']) && is_numeric($_GET['author'])) {
            wp_redirect(home_url(), 301);
            exit;
        }
    }

    public function disable_rest_users($endpoints) {
        if (!current_user_can('list_users')) {
            unset($endpoints['/wp/v2/users']);
            unset($endpoints['/wp/v2/users/(?P<id>[\d]+)']);
        }
        return $endpoints;
    }

    public function get_status() {
        return array(
            'hide_wp_version' => array(
                'label' => 'WordPress-Version verbergen',
                'enabled' => (bool) get_option('safetypress_hide_wp_version', 1),
                'description' => 'Entfernt WordPress-Versionsinformationen.',
            ),
            'disable_xmlrpc' => array(
                'label' => 'XML-RPC deaktivieren',
                'enabled' => (bool) get_option('safetypress_disable_xmlrpc', 1),
                'description' => 'Deaktiviert XML-RPC.',
            ),
            'disable_file_editor' => array(
                'label' => 'Datei-Editor deaktivieren',
                'enabled' => (bool) get_option('safetypress_disable_file_editor', 1),
                'description' => 'Deaktiviert den Theme- und Plugin-Editor.',
            ),
            'security_headers' => array(
                'label' => 'Sicherheits-Header',
                'enabled' => true,
                'description' => 'X-Content-Type, X-Frame-Options, X-XSS-Protection, Referrer-Policy, Permissions-Policy',
            ),
            'ssl_enabled' => array(
                'label' => 'SSL/HTTPS',
                'enabled' => is_ssl(),
                'description' => 'Verschlüsselte Verbindung.',
            ),
            'csp_enabled' => array(
                'label' => 'Content-Security-Policy',
                'enabled' => (bool) get_option('safetypress_enable_csp', 0),
                'description' => 'Schutz vor XSS und Injection-Angriffen.',
            ),
        );
    }
    
    /**
     * Aktive Security Headers anzeigen
     */
    public function get_active_headers() {
        $headers = [
            'X-Content-Type-Options' => 'nosniff',
            'X-Frame-Options' => 'SAMEORIGIN',
            'X-XSS-Protection' => '1; mode=block',
            'Referrer-Policy' => 'strict-origin-when-cross-origin',
            'Permissions-Policy' => 'camera=(), microphone=(), geolocation=()...',
        ];
        
        if (is_ssl()) {
            $headers['Strict-Transport-Security'] = 'max-age=31536000; includeSubDomains';
        }
        
        if (get_option('safetypress_enable_csp', 0)) {
            $headers['Content-Security-Policy'] = get_option('safetypress_csp_mode', 'basic') . ' mode';
        }
        
        return $headers;
    }
}
