<?php
if (!defined('ABSPATH')) exit;

class SafetyPress_Firewall {

    private $blocked_table;
    private $cache_key = 'safetypress_fw_cache';
    private $rate_limit_prefix = 'safetypress_rate_';

    public function __construct() {
        global $wpdb;
        $this->blocked_table = $wpdb->prefix . 'safetypress_blocked_ips';
        
        if (get_option('safetypress_enable_firewall', 1)) {
            add_action('init', array($this, 'check_request'), 1);
            
            // PRO Features
            if ($this->is_pro()) {
                // Rate Limiting
                if (get_option('safetypress_fw_rate_limiting', 0)) {
                    add_action('init', array($this, 'check_rate_limit'), 0);
                }
                
                // Antispam
                if (get_option('safetypress_fw_antispam', 0)) {
                    add_filter('preprocess_comment', array($this, 'check_comment_spam'), 1);
                    add_filter('registration_errors', array($this, 'check_registration_spam'), 10, 3);
                }
                
                // Response Body Filter
                if (get_option('safetypress_fw_response_filter', 0)) {
                    add_action('template_redirect', array($this, 'start_output_buffer'), 0);
                }
                
                // File Upload Filter
                if (get_option('safetypress_fw_upload_filter', 1)) {
                    add_filter('wp_handle_upload_prefilter', array($this, 'check_file_upload'));
                }
            }
        }
        
        // AJAX Handlers
        add_action('wp_ajax_safetypress_block_ip', array($this, 'ajax_block_ip'));
        add_action('wp_ajax_safetypress_unblock_ip', array($this, 'ajax_unblock_ip'));
        add_action('wp_ajax_safetypress_save_firewall_rules', array($this, 'ajax_save_rules'));
        add_action('wp_ajax_safetypress_test_ip_country', array($this, 'ajax_test_ip_country'));
    }

    private function is_pro() {
        return function_exists('safetypress') && safetypress()->is_pro();
    }

    /**
     * Haupt-Request-Prüfung
     */
    public function check_request() {
        $ip = SafetyPress_Helper::get_client_ip();
        
        // Whitelist Check (immer zuerst)
        if ($this->is_whitelisted($ip)) return;
        
        // Blocked IP Check
        if ($this->is_blocked($ip)) {
            $this->log_event('firewall_blocked', sprintf('Blockierte IP %s versuchte Zugriff', $ip));
            $this->block_response('IP blockiert');
        }
        
        // PRO: Country-based Blocking
        if ($this->is_pro() && get_option('safetypress_fw_country_blocking', 0)) {
            if ($this->is_country_blocked($ip)) {
                $this->log_event('country_blocked', sprintf('IP %s aus blockiertem Land', $ip));
                $this->block_response('Zugriff aus Ihrem Land nicht erlaubt');
            }
        }
        
        // PRO: URL-based Blocking
        if ($this->is_pro() && get_option('safetypress_fw_url_blocking', 0)) {
            if ($this->is_url_blocked()) {
                $this->log_event('url_blocked', sprintf('Blockierte URL von IP %s', $ip));
                $this->block_response('URL nicht erlaubt');
            }
        }
        
        // PRO: Bot Blocking
        if ($this->is_pro() && get_option('safetypress_fw_bot_blocking', 0)) {
            if ($this->is_bot_blocked()) {
                $this->log_event('bot_blocked', sprintf('Blockierter Bot von IP %s', $ip));
                $this->block_response('Bot nicht erlaubt');
            }
        }
        
        // PRO: Role-based Access
        if ($this->is_pro() && get_option('safetypress_fw_role_access', 0)) {
            if (!$this->check_role_access()) {
                $this->log_event('role_blocked', sprintf('Rollenzugriff verweigert für IP %s', $ip));
                $this->block_response('Zugriff für Ihre Rolle nicht erlaubt');
            }
        }
        
        // Malicious Request Check (Free)
        if ($this->is_malicious()) {
            $this->handle_malicious($ip);
        }
        
        // Bad Bot Check (Free - Basis)
        if (SafetyPress_Helper::is_bad_bot()) {
            $this->log_event('bad_bot_blocked', sprintf('Bad Bot blockiert von IP %s', $ip));
            $this->block_response('Bot nicht erlaubt');
        }
    }

    /**
     * ==========================================
     * IP ACCESS CONTROL
     * ==========================================
     */
    
    public function is_whitelisted($ip) {
        // Standard Whitelist
        $default_whitelist = array('127.0.0.1', '::1');
        
        // PRO: Custom Whitelist
        if ($this->is_pro()) {
            $custom_whitelist = get_option('safetypress_fw_ip_whitelist', '');
            if (!empty($custom_whitelist)) {
                $custom_ips = array_map('trim', explode("\n", $custom_whitelist));
                $default_whitelist = array_merge($default_whitelist, $custom_ips);
            }
        }
        
        // IP-Range Check
        foreach ($default_whitelist as $allowed_ip) {
            if ($this->ip_matches($ip, $allowed_ip)) {
                return true;
            }
        }
        
        return apply_filters('safetypress_ip_whitelist_check', false, $ip);
    }

    /**
     * Prüft ob IP in Range/CIDR liegt
     */
    private function ip_matches($ip, $pattern) {
        // Exact match
        if ($ip === $pattern) return true;
        
        // CIDR notation (e.g., 192.168.1.0/24)
        if (strpos($pattern, '/') !== false) {
            list($subnet, $mask) = explode('/', $pattern);
            $subnet_long = ip2long($subnet);
            $ip_long = ip2long($ip);
            $mask_long = -1 << (32 - intval($mask));
            
            return ($ip_long & $mask_long) === ($subnet_long & $mask_long);
        }
        
        // Wildcard (e.g., 192.168.1.*)
        if (strpos($pattern, '*') !== false) {
            $regex = str_replace('.', '\.', $pattern);
            $regex = str_replace('*', '\d+', $regex);
            return preg_match('/^' . $regex . '$/', $ip);
        }
        
        return false;
    }

    /**
     * ==========================================
     * COUNTRY-BASED ACCESS CONTROL (PRO)
     * ==========================================
     */
    
    public function is_country_blocked($ip) {
        $blocked_countries = get_option('safetypress_fw_blocked_countries', array());
        if (empty($blocked_countries)) return false;
        
        $country = $this->get_country_code($ip);
        if ($country === 'XX') return false; // Unknown
        
        return in_array($country, $blocked_countries);
    }

    public function get_country_code($ip) {
        // Cache check
        $cache_key = 'safetypress_geo_' . md5($ip);
        $cached = get_transient($cache_key);
        if ($cached !== false) return $cached;
        
        // API Call
        $response = wp_remote_get("http://ip-api.com/json/{$ip}?fields=countryCode", array(
            'timeout' => 3,
            'sslverify' => false
        ));
        
        $country = 'XX';
        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            if (isset($data['countryCode'])) {
                $country = $data['countryCode'];
            }
        }
        
        set_transient($cache_key, $country, DAY_IN_SECONDS);
        return $country;
    }

    public function get_all_countries() {
        return array(
            'AF' => 'Afghanistan', 'AL' => 'Albanien', 'DZ' => 'Algerien', 'AD' => 'Andorra',
            'AO' => 'Angola', 'AR' => 'Argentinien', 'AM' => 'Armenien', 'AU' => 'Australien',
            'AT' => 'Österreich', 'AZ' => 'Aserbaidschan', 'BD' => 'Bangladesch', 'BY' => 'Belarus',
            'BE' => 'Belgien', 'BA' => 'Bosnien-Herzegowina', 'BR' => 'Brasilien', 'BG' => 'Bulgarien',
            'CA' => 'Kanada', 'CL' => 'Chile', 'CN' => 'China', 'CO' => 'Kolumbien',
            'HR' => 'Kroatien', 'CU' => 'Kuba', 'CY' => 'Zypern', 'CZ' => 'Tschechien',
            'DK' => 'Dänemark', 'EC' => 'Ecuador', 'EG' => 'Ägypten', 'EE' => 'Estland',
            'FI' => 'Finnland', 'FR' => 'Frankreich', 'GE' => 'Georgien', 'DE' => 'Deutschland',
            'GR' => 'Griechenland', 'HK' => 'Hongkong', 'HU' => 'Ungarn', 'IS' => 'Island',
            'IN' => 'Indien', 'ID' => 'Indonesien', 'IR' => 'Iran', 'IQ' => 'Irak',
            'IE' => 'Irland', 'IL' => 'Israel', 'IT' => 'Italien', 'JP' => 'Japan',
            'KZ' => 'Kasachstan', 'KE' => 'Kenia', 'KP' => 'Nordkorea', 'KR' => 'Südkorea',
            'KW' => 'Kuwait', 'LV' => 'Lettland', 'LB' => 'Libanon', 'LT' => 'Litauen',
            'LU' => 'Luxemburg', 'MY' => 'Malaysia', 'MX' => 'Mexiko', 'MD' => 'Moldawien',
            'MA' => 'Marokko', 'NL' => 'Niederlande', 'NZ' => 'Neuseeland', 'NG' => 'Nigeria',
            'NO' => 'Norwegen', 'PK' => 'Pakistan', 'PA' => 'Panama', 'PE' => 'Peru',
            'PH' => 'Philippinen', 'PL' => 'Polen', 'PT' => 'Portugal', 'QA' => 'Katar',
            'RO' => 'Rumänien', 'RU' => 'Russland', 'SA' => 'Saudi-Arabien', 'RS' => 'Serbien',
            'SG' => 'Singapur', 'SK' => 'Slowakei', 'SI' => 'Slowenien', 'ZA' => 'Südafrika',
            'ES' => 'Spanien', 'SE' => 'Schweden', 'CH' => 'Schweiz', 'SY' => 'Syrien',
            'TW' => 'Taiwan', 'TH' => 'Thailand', 'TR' => 'Türkei', 'UA' => 'Ukraine',
            'AE' => 'VAE', 'GB' => 'Großbritannien', 'US' => 'USA', 'VE' => 'Venezuela',
            'VN' => 'Vietnam',
        );
    }

    /**
     * ==========================================
     * URL-BASED ACCESS CONTROL (PRO)
     * ==========================================
     */
    
    public function is_url_blocked() {
        $blocked_urls = get_option('safetypress_fw_blocked_urls', '');
        if (empty($blocked_urls)) return false;
        
        $current_url = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '';
        $patterns = array_filter(array_map('trim', explode("\n", $blocked_urls)));
        
        foreach ($patterns as $pattern) {
            // Regex pattern (starts with /)
            if (substr($pattern, 0, 1) === '/' && substr($pattern, -1) === '/') {
                if (@preg_match($pattern, $current_url)) return true;
            }
            // Wildcard pattern
            elseif (strpos($pattern, '*') !== false) {
                $regex = str_replace('*', '.*', preg_quote($pattern, '/'));
                if (preg_match('/^' . $regex . '$/i', $current_url)) return true;
            }
            // Exact match
            elseif (stripos($current_url, $pattern) !== false) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * ==========================================
     * BOT-BASED ACCESS CONTROL (PRO)
     * ==========================================
     */
    
    public function is_bot_blocked() {
        $blocked_bots = get_option('safetypress_fw_blocked_bots', '');
        if (empty($blocked_bots)) return false;
        
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? strtolower($_SERVER['HTTP_USER_AGENT']) : '';
        if (empty($user_agent)) return get_option('safetypress_fw_block_empty_ua', 0);
        
        $patterns = array_filter(array_map('trim', explode("\n", strtolower($blocked_bots))));
        
        foreach ($patterns as $pattern) {
            if (strpos($user_agent, $pattern) !== false) {
                return true;
            }
        }
        
        return false;
    }

    public function get_default_bad_bots() {
        return "ahrefsbot\nsemrushbot\ndotbot\nmj12bot\nblexbot\npetalbot\nbytespider\nclaudebot\ngptbot\nccbot\nscreaming frog\nnetcraftsurvey\nsqlmap\nnikto\nnessus\nacunetix\nburpsuite\ndirbuster\nwpscan\nhavij";
    }

    /**
     * ==========================================
     * ROLE-BASED ACCESS CONTROL (PRO)
     * ==========================================
     */
    
    public function check_role_access() {
        // Nur für eingeloggte Benutzer relevant
        if (!is_user_logged_in()) return true;
        
        $restricted_areas = get_option('safetypress_fw_role_restrictions', array());
        if (empty($restricted_areas)) return true;
        
        $current_url = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '';
        $user = wp_get_current_user();
        $user_roles = $user->roles;
        
        foreach ($restricted_areas as $restriction) {
            // Format: "url_pattern|allowed_roles" (comma-separated)
            if (strpos($current_url, $restriction['url']) !== false) {
                $allowed_roles = array_map('trim', explode(',', $restriction['roles']));
                $has_access = false;
                
                foreach ($user_roles as $role) {
                    if (in_array($role, $allowed_roles)) {
                        $has_access = true;
                        break;
                    }
                }
                
                if (!$has_access) return false;
            }
        }
        
        return true;
    }

    /**
     * ==========================================
     * RATE LIMITING (PRO)
     * ==========================================
     */
    
    public function check_rate_limit() {
        $ip = SafetyPress_Helper::get_client_ip();
        if ($this->is_whitelisted($ip)) return;
        
        $limit = intval(get_option('safetypress_fw_rate_limit', 60)); // Requests per minute
        $window = 60; // 1 minute window
        
        $key = $this->rate_limit_prefix . md5($ip);
        $current = get_transient($key);
        
        if ($current === false) {
            set_transient($key, 1, $window);
            return;
        }
        
        if ($current >= $limit) {
            $this->log_event('rate_limited', sprintf('IP %s hat Rate-Limit überschritten (%d/%d)', $ip, $current, $limit));
            
            // Auto-Block wenn zu oft überschritten
            $block_key = $this->rate_limit_prefix . 'block_' . md5($ip);
            $block_count = get_transient($block_key);
            
            if ($block_count >= 3) {
                $this->block_ip($ip, 'Auto-Block: Rate-Limit wiederholt überschritten', HOUR_IN_SECONDS);
            } else {
                set_transient($block_key, ($block_count ? $block_count + 1 : 1), HOUR_IN_SECONDS);
            }
            
            $this->block_response('Zu viele Anfragen. Bitte warten Sie.', 429);
        }
        
        set_transient($key, $current + 1, $window);
    }

    /**
     * ==========================================
     * ANTISPAM (PRO)
     * ==========================================
     */
    
    public function check_comment_spam($commentdata) {
        $ip = SafetyPress_Helper::get_client_ip();
        
        // Honeypot Check
        if (!empty($_POST['safetypress_hp_field'])) {
            $this->log_event('spam_comment_blocked', sprintf('Honeypot ausgelöst von IP %s', $ip));
            wp_die('Spam erkannt.', 'Spam', array('response' => 403));
        }
        
        // Time-based Check (zu schnell ausgefüllt)
        $form_time = isset($_POST['safetypress_form_time']) ? intval($_POST['safetypress_form_time']) : 0;
        if ($form_time > 0 && (time() - $form_time) < 3) {
            $this->log_event('spam_comment_blocked', sprintf('Zu schnell ausgefüllt von IP %s', $ip));
            wp_die('Bitte langsamer ausfüllen.', 'Spam', array('response' => 403));
        }
        
        // Link Count Check
        $max_links = intval(get_option('safetypress_fw_spam_max_links', 3));
        $content = isset($commentdata['comment_content']) ? $commentdata['comment_content'] : '';
        $link_count = preg_match_all('/<a\s|https?:\/\//i', $content);
        
        if ($link_count > $max_links) {
            $this->log_event('spam_comment_blocked', sprintf('Zu viele Links (%d) von IP %s', $link_count, $ip));
            wp_die('Zu viele Links im Kommentar.', 'Spam', array('response' => 403));
        }
        
        // Blacklist Words
        $blacklist = get_option('safetypress_fw_spam_blacklist', '');
        if (!empty($blacklist)) {
            $words = array_filter(array_map('trim', explode("\n", strtolower($blacklist))));
            $content_lower = strtolower($content);
            
            foreach ($words as $word) {
                if (strpos($content_lower, $word) !== false) {
                    $this->log_event('spam_comment_blocked', sprintf('Blacklist-Wort gefunden von IP %s', $ip));
                    wp_die('Unerlaubter Inhalt erkannt.', 'Spam', array('response' => 403));
                }
            }
        }
        
        return $commentdata;
    }

    public function check_registration_spam($errors, $sanitized_user_login, $user_email) {
        $ip = SafetyPress_Helper::get_client_ip();
        
        // Honeypot Check
        if (!empty($_POST['safetypress_hp_field'])) {
            $errors->add('spam', 'Spam erkannt.');
            $this->log_event('spam_registration_blocked', sprintf('Honeypot ausgelöst von IP %s', $ip));
        }
        
        // Disposable Email Check
        if (get_option('safetypress_fw_block_disposable_email', 1)) {
            $disposable_domains = array(
                'mailinator.com', 'guerrillamail.com', 'tempmail.com', '10minutemail.com',
                'throwaway.email', 'fakeinbox.com', 'trashmail.com', 'maildrop.cc',
                'yopmail.com', 'sharklasers.com', 'getnada.com', 'temp-mail.org'
            );
            
            $email_domain = strtolower(substr(strrchr($user_email, '@'), 1));
            if (in_array($email_domain, $disposable_domains)) {
                $errors->add('disposable_email', 'Einweg-E-Mail-Adressen sind nicht erlaubt.');
                $this->log_event('spam_registration_blocked', sprintf('Einweg-E-Mail von IP %s: %s', $ip, $user_email));
            }
        }
        
        return $errors;
    }

    /**
     * ==========================================
     * RESPONSE BODY FILTER (PRO)
     * ==========================================
     */
    
    public function start_output_buffer() {
        if (is_admin()) return;
        ob_start(array($this, 'filter_response_body'));
    }

    public function filter_response_body($buffer) {
        // Response filtering disabled to ensure compatibility with hosting providers
        // Patterns would trigger false positives in security scanners
        return $buffer;
    }

    /**
     * ==========================================
     * FILE UPLOAD FILTER (PRO)
     * ==========================================
     */
    
    public function check_file_upload($file) {
        $ip = SafetyPress_Helper::get_client_ip();
        $filename = isset($file['name']) ? strtolower($file['name']) : '';
        $extension = pathinfo($filename, PATHINFO_EXTENSION);
        
        // Gefährliche Erweiterungen aus Einstellungen laden
        $blocked_ext_string = get_option('safetypress_fw_blocked_extensions', '');
        $dangerous_extensions = array_filter(array_map('trim', explode("\n", $blocked_ext_string)));
        
        // Falls leer, Standard-Liste verwenden
        if (empty($dangerous_extensions)) {
            $dangerous_extensions = array('phtml', 'php3', 'php4', 'php5');
        }
        
        // Extension check
        if (in_array($extension, $dangerous_extensions)) {
            $file['error'] = 'Dateityp nicht erlaubt.';
            $this->log_event('upload_blocked', sprintf('Gefährliche Erweiterung blockiert von IP %s: %s', $ip, $filename));
            return $file;
        }
        
        // Double extension check
        foreach ($dangerous_extensions as $ext) {
            if (strpos($filename, '.' . $ext . '.') !== false) {
                $file['error'] = 'Dateityp nicht erlaubt (versteckte Erweiterung).';
                $this->log_event('upload_blocked', sprintf('Versteckte Erweiterung blockiert von IP %s: %s', $ip, $filename));
                return $file;
            }
        }
        
        // Max file size (custom limit)
        $max_size = intval(get_option('safetypress_fw_max_upload_size', 10)) * 1024 * 1024; // MB to bytes
        if (isset($file['size']) && $file['size'] > $max_size) {
            $file['error'] = sprintf('Datei zu groß (max. %d MB).', get_option('safetypress_fw_max_upload_size', 10));
            return $file;
        }
        
        return $file;
    }

    /**
     * ==========================================
     * SYSLOG LOGGING (PRO)
     * ==========================================
     */
    
    public function syslog_write($message, $priority = LOG_WARNING) {
        if (!get_option('safetypress_fw_syslog', 0)) return;
        
        $facility = intval(get_option('safetypress_fw_syslog_facility', LOG_LOCAL0));
        
        openlog('SafetyPress', LOG_PID | LOG_PERROR, $facility);
        syslog($priority, $message);
        closelog();
    }

    /**
     * ==========================================
     * STANDARD FIREWALL FUNCTIONS
     * ==========================================
     */

    private function is_malicious() {
        $check = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '';
        $check .= isset($_SERVER['QUERY_STRING']) ? $_SERVER['QUERY_STRING'] : '';
        
        // Patterns werden von der API geladen (gecached)
        $patterns = $this->get_malicious_patterns();
        
        if (empty($patterns)) {
            return false;
        }
        
        foreach ($patterns as $p) {
            if (@preg_match($p, $check)) return true;
        }
        
        // POST data check
        if (!empty($_POST)) {
            $post_data = json_encode($_POST);
            foreach ($patterns as $p) {
                if (@preg_match($p, $post_data)) return true;
            }
        }
        
        return false;
    }
    
    /**
     * Lädt Firewall-Patterns von der API (gecached)
     */
    private function get_malicious_patterns() {
        $cache_key = 'safetypress_fw_patterns';
        $patterns = get_transient($cache_key);
        
        if ($patterns !== false) {
            return $patterns;
        }
        
        // Von API laden
        $response = wp_remote_get('https://safetypress.de/api/license.php?action=get_fw_patterns', [
            'timeout' => 5,
            'sslverify' => true,
        ]);
        
        if (is_wp_error($response)) {
            return [];
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        $patterns = isset($data['patterns']) ? $data['patterns'] : [];
        
        // 24 Stunden cachen
        set_transient($cache_key, $patterns, DAY_IN_SECONDS);
        
        return $patterns;
    }

    private function handle_malicious($ip) {
        $key = 'safetypress_violations_' . md5($ip);
        $v = get_transient($key);
        $v = $v ? $v + 1 : 1;
        set_transient($key, $v, HOUR_IN_SECONDS);
        
        $this->log_event('malicious_request', sprintf('Verdächtige Anfrage von IP %s blockiert (Versuch %d)', $ip, $v), array(
            'request_uri' => isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '',
        ));
        
        if ($v >= 5) {
            $this->block_ip($ip, 'Automatisch blockiert - Wiederholte Angriffe', HOUR_IN_SECONDS);
            $this->log_event('ip_auto_blocked', sprintf('IP %s automatisch für 1 Stunde blockiert nach %d Angriffen', $ip, $v));
        }
        
        $this->block_response();
    }

    public function block_ip($ip, $reason = '', $duration = null) {
        global $wpdb;
        $ip = filter_var($ip, FILTER_VALIDATE_IP);
        if (!$ip) return false;
        
        $result = $wpdb->replace($this->blocked_table, array(
            'ip_address' => $ip,
            'reason' => sanitize_text_field($reason),
            'blocked_until' => $duration ? date('Y-m-d H:i:s', time() + $duration) : null,
            'is_permanent' => $duration ? 0 : 1,
            'created_at' => current_time('mysql'),
        ));
        
        // Syslog
        if ($this->is_pro()) {
            $this->syslog_write(sprintf('IP blocked: %s - Reason: %s', $ip, $reason), LOG_WARNING);
        }
        
        return $result !== false;
    }

    public function unblock_ip($ip) {
        global $wpdb;
        return $wpdb->delete($this->blocked_table, array('ip_address' => $ip));
    }

    public function is_blocked($ip) {
        global $wpdb;
        $blocked = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->blocked_table} WHERE ip_address = %s", $ip
        ));
        if (!$blocked) return false;
        if ($blocked->is_permanent) return true;
        if ($blocked->blocked_until && strtotime($blocked->blocked_until) > time()) return true;
        $this->unblock_ip($ip);
        return false;
    }

    public function get_blocked_ips() {
        global $wpdb;
        return $wpdb->get_results("SELECT * FROM {$this->blocked_table} ORDER BY created_at DESC");
    }

    private function block_response($message = 'Zugriff verweigert', $code = 403) {
        status_header($code);
        
        // Custom block page
        $custom_page = get_option('safetypress_fw_block_page', '');
        if (!empty($custom_page)) {
            echo wp_kses_post($custom_page);
            exit;
        }
        
        wp_die(
            '<h1>🛡️ SafetyPress</h1><p>' . esc_html($message) . '</p>',
            'Blockiert',
            array('response' => $code)
        );
    }
    
    private function log_event($type, $message, $extra = array()) {
        if (function_exists('safetypress') && safetypress()->logger) {
            safetypress()->logger->log($type, $message, $extra);
        }
        
        // Syslog
        if ($this->is_pro()) {
            $this->syslog_write($message);
        }
    }

    /**
     * ==========================================
     * AJAX HANDLERS
     * ==========================================
     */

    public function ajax_block_ip() {
        check_ajax_referer('safetypress_admin', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        
        $ip = isset($_POST['ip']) ? sanitize_text_field($_POST['ip']) : '';
        $reason = isset($_POST['reason']) ? sanitize_text_field($_POST['reason']) : 'Manuell blockiert';
        $permanent = isset($_POST['permanent']) && $_POST['permanent'] === 'true';
        
        if (!filter_var($ip, FILTER_VALIDATE_IP)) wp_send_json_error(array('message' => 'Ungültige IP.'));
        
        if ($this->block_ip($ip, $reason, $permanent ? null : DAY_IN_SECONDS)) {
            $this->log_event('ip_manual_block', sprintf('IP %s manuell blockiert: %s', $ip, $reason), array('permanent' => $permanent));
            wp_send_json_success(array('message' => 'IP blockiert.'));
        }
        wp_send_json_error(array('message' => 'Fehler.'));
    }

    public function ajax_unblock_ip() {
        check_ajax_referer('safetypress_admin', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        
        $ip = isset($_POST['ip']) ? sanitize_text_field($_POST['ip']) : '';
        if ($this->unblock_ip($ip)) {
            $this->log_event('ip_unblocked', sprintf('IP %s wurde entsperrt', $ip));
            wp_send_json_success(array('message' => 'IP entsperrt.'));
        }
        wp_send_json_error(array('message' => 'Fehler.'));
    }

    public function ajax_save_rules() {
        check_ajax_referer('safetypress_admin', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        
        $settings = isset($_POST['settings']) ? $_POST['settings'] : array();
        
        foreach ($settings as $key => $value) {
            $option_name = 'safetypress_fw_' . sanitize_key($key);
            
            if (is_array($value)) {
                update_option($option_name, array_map('sanitize_text_field', $value));
            } else {
                update_option($option_name, sanitize_textarea_field($value));
            }
        }
        
        $this->log_event('firewall_rules_updated', 'Firewall-Regeln wurden aktualisiert');
        wp_send_json_success(array('message' => 'Einstellungen gespeichert.'));
    }

    public function ajax_test_ip_country() {
        check_ajax_referer('safetypress_admin', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(array('message' => 'Keine Berechtigung.'));
        
        $ip = isset($_POST['ip']) ? sanitize_text_field($_POST['ip']) : '';
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            wp_send_json_error(array('message' => 'Ungültige IP.'));
        }
        
        $country = $this->get_country_code($ip);
        $countries = $this->get_all_countries();
        $country_name = isset($countries[$country]) ? $countries[$country] : 'Unbekannt';
        
        wp_send_json_success(array(
            'country_code' => $country,
            'country_name' => $country_name,
        ));
    }

    /**
     * Statistiken für Dashboard
     */
    public function get_statistics() {
        global $wpdb;
        
        $blocked_count = $wpdb->get_var("SELECT COUNT(*) FROM {$this->blocked_table}");
        $permanent_count = $wpdb->get_var("SELECT COUNT(*) FROM {$this->blocked_table} WHERE is_permanent = 1");
        
        return array(
            'total_blocked' => intval($blocked_count),
            'permanent_blocked' => intval($permanent_count),
            'temporary_blocked' => intval($blocked_count) - intval($permanent_count),
        );
    }
}
