<?php
if (!defined('ABSPATH')) exit;

class SafetyPress_Admin {

    public function __construct() {
        add_action('admin_menu', array($this, 'add_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_init', array($this, 'register_settings'));
        
        // AJAX Handlers
        add_action('wp_ajax_safetypress_export_logs', array($this, 'export_logs'));
        add_action('wp_ajax_safetypress_send_test_email', array($this, 'ajax_send_test_email'));
        add_action('wp_ajax_safetypress_get_security_score', array($this, 'ajax_get_security_score'));
        
        // Settings-Änderungen loggen
        add_action('update_option', array($this, 'log_settings_change'), 10, 3);
    }

    public function add_menu() {
        add_menu_page('SafetyPress', 'Security', 'manage_options', 'safetypress', array($this, 'page_dashboard'), 'dashicons-shield-alt', 80);
        add_submenu_page('safetypress', 'Dashboard', 'Dashboard', 'manage_options', 'safetypress', array($this, 'page_dashboard'));
        add_submenu_page('safetypress', 'Firewall', 'Firewall', 'manage_options', 'safetypress-firewall', array($this, 'page_firewall'));
        add_submenu_page('safetypress', 'Login-Schutz', 'Login-Schutz', 'manage_options', 'safetypress-login', array($this, 'page_login'));
        add_submenu_page('safetypress', 'Hardening', 'Hardening', 'manage_options', 'safetypress-hardening', array($this, 'page_hardening'));
        add_submenu_page('safetypress', 'Scanner', 'Scanner', 'manage_options', 'safetypress-scanner', array($this, 'page_scanner'));
        add_submenu_page('safetypress', 'Integrität', 'Integrität', 'manage_options', 'safetypress-integrity', array($this, 'page_integrity'));
        add_submenu_page('safetypress', 'Activity Log', 'Activity Log', 'manage_options', 'safetypress-logs', array($this, 'page_logs'));
        add_submenu_page('safetypress', 'Einstellungen', 'Einstellungen', 'manage_options', 'safetypress-settings', array($this, 'page_settings'));
        add_submenu_page('safetypress', 'Lizenz', 'Lizenz', 'manage_options', 'safetypress-license', array($this, 'page_license'));
    }

    public function enqueue_scripts($hook) {
        if (strpos($hook, 'safetypress') === false) return;
        
        wp_enqueue_style('safetypress-admin', SAFETYPRESS_PLUGIN_URL . 'assets/css/admin-style.css', array(), SAFETYPRESS_VERSION);
        wp_enqueue_script('safetypress-admin', SAFETYPRESS_PLUGIN_URL . 'assets/js/admin-script.js', array('jquery'), SAFETYPRESS_VERSION, true);
        wp_localize_script('safetypress-admin', 'safetypress', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('safetypress_admin'),
        ));
    }

    public function sanitize_checkbox($value) {
        return $value ? 1 : 0;
    }

    public function sanitize_integer($value) {
        return absint($value);
    }

    public function register_settings() {
        // Allgemeine Einstellungen
        $general_checkboxes = array(
            'safetypress_enable_firewall',
            'safetypress_enable_login_protection', 
            'safetypress_enable_hardening',
            'safetypress_enable_activity_log',
            'safetypress_enable_woo_protection',
            'safetypress_email_notifications',
            'safetypress_enable_2fa',
            'safetypress_scheduled_scan_email',
        );
        foreach ($general_checkboxes as $opt) {
            register_setting('safetypress_general', $opt, array(
                'type' => 'integer',
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => 1,
            ));
        }
        
        // Custom Login separat mit default 0
        register_setting('safetypress_general', 'safetypress_enable_custom_login', array(
            'type' => 'integer',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => 0,
        ));
        
        register_setting('safetypress_general', 'safetypress_fraud_score_threshold', array(
            'type' => 'integer',
            'sanitize_callback' => array($this, 'sanitize_integer'),
            'default' => 70,
        ));
        register_setting('safetypress_general', 'safetypress_log_retention_days', array(
            'type' => 'integer',
            'sanitize_callback' => array($this, 'sanitize_integer'),
            'default' => 30,
        ));
        register_setting('safetypress_general', 'safetypress_notification_email', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_email',
            'default' => get_option('admin_email'),
        ));
        register_setting('safetypress_general', 'safetypress_scheduled_scan', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'disabled',
        ));
        register_setting('safetypress_general', 'safetypress_scheduled_scan_time', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => '03:00',
        ));
        register_setting('safetypress_general', 'safetypress_custom_login_url', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_title',
            'default' => '',
        ));
        register_setting('safetypress_general', 'safetypress_login_redirect_url', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => '',
        ));

        // Hardening-Einstellungen
        $hardening_checkboxes = array(
            'safetypress_hide_wp_version',
            'safetypress_disable_xmlrpc',
            'safetypress_disable_file_editor',
        );
        foreach ($hardening_checkboxes as $opt) {
            register_setting('safetypress_hardening', $opt, array(
                'type' => 'integer',
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => 1,
            ));
        }

        // Login-Schutz-Einstellungen
        register_setting('safetypress_login', 'safetypress_enable_login_protection', array(
            'type' => 'integer',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => 1,
        ));
        register_setting('safetypress_login', 'safetypress_max_login_attempts', array(
            'type' => 'integer',
            'sanitize_callback' => array($this, 'sanitize_integer'),
            'default' => 5,
        ));
        register_setting('safetypress_login', 'safetypress_lockout_duration', array(
            'type' => 'integer',
            'sanitize_callback' => array($this, 'sanitize_integer'),
            'default' => 30,
        ));

        // Firewall-Einstellungen
        register_setting('safetypress_firewall', 'safetypress_enable_firewall', array(
            'type' => 'integer',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => 1,
        ));
    }

    // Page Methods
    public function page_dashboard() { include SAFETYPRESS_PLUGIN_DIR . 'admin/views/dashboard.php'; }
    public function page_firewall() { include SAFETYPRESS_PLUGIN_DIR . 'admin/views/firewall.php'; }
    public function page_login() { include SAFETYPRESS_PLUGIN_DIR . 'admin/views/login-protection.php'; }
    public function page_hardening() { include SAFETYPRESS_PLUGIN_DIR . 'admin/views/hardening.php'; }
    public function page_scanner() { include SAFETYPRESS_PLUGIN_DIR . 'admin/views/scanner.php'; }
    public function page_integrity() { include SAFETYPRESS_PLUGIN_DIR . 'admin/views/integrity.php'; }
    public function page_logs() { include SAFETYPRESS_PLUGIN_DIR . 'admin/views/logs.php'; }
    public function page_settings() { include SAFETYPRESS_PLUGIN_DIR . 'admin/views/settings.php'; }
    public function page_license() { include SAFETYPRESS_PLUGIN_DIR . 'admin/views/license.php'; }

    // AJAX: Export Logs
    public function export_logs() {
        check_ajax_referer('safetypress_admin', 'nonce');
        if (!current_user_can('manage_options')) wp_die('Keine Berechtigung.');
        $logs = safetypress()->logger->get_logs(array('limit' => 10000));
        $csv = safetypress()->logger->export_csv($logs);
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="safetypress-logs-' . date('Y-m-d') . '.csv"');
        echo $csv;
        exit;
    }
    
    // AJAX: Test-E-Mail senden
    public function ajax_send_test_email() {
        check_ajax_referer('safetypress_admin', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Keine Berechtigung']);
        }
        
        if (safetypress()->notifications) {
            $result = safetypress()->notifications->send_test_email();
            if ($result) {
                wp_send_json_success(['message' => 'Test-E-Mail gesendet']);
            }
        }
        
        wp_send_json_error(['message' => 'E-Mail konnte nicht gesendet werden']);
    }
    
    // AJAX: Security Score
    public function ajax_get_security_score() {
        check_ajax_referer('safetypress_admin', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Keine Berechtigung']);
        }
        
        if (safetypress()->security_score) {
            $result = safetypress()->security_score->calculate_score();
            wp_send_json_success($result);
        }
        
        wp_send_json_error(['message' => 'Score konnte nicht berechnet werden']);
    }
    
    /**
     * Loggt Änderungen an SafetyPress-Einstellungen
     */
    public function log_settings_change($option, $old_value, $new_value) {
        if (strpos($option, 'safetypress_') !== 0) return;
        if ($old_value === $new_value) return;
        if (!function_exists('safetypress') || !safetypress()->logger) return;
        
        $labels = array(
            'safetypress_enable_firewall' => 'Firewall',
            'safetypress_enable_login_protection' => 'Login-Schutz',
            'safetypress_enable_hardening' => 'Hardening',
            'safetypress_enable_activity_log' => 'Activity Log',
            'safetypress_enable_woo_protection' => 'WooCommerce Schutz',
            'safetypress_email_notifications' => 'E-Mail-Benachrichtigungen',
            'safetypress_enable_2fa' => 'Zwei-Faktor-Authentifizierung',
            'safetypress_hide_wp_version' => 'WP-Version verbergen',
            'safetypress_disable_xmlrpc' => 'XML-RPC deaktivieren',
            'safetypress_disable_file_editor' => 'Datei-Editor deaktivieren',
            'safetypress_max_login_attempts' => 'Max. Login-Versuche',
            'safetypress_lockout_duration' => 'Sperrdauer',
            'safetypress_fraud_score_threshold' => 'Fraud-Score Schwelle',
            'safetypress_log_retention_days' => 'Log-Aufbewahrung',
            'safetypress_notification_email' => 'Benachrichtigungs-E-Mail',
            'safetypress_scheduled_scan' => 'Geplante Scans',
            'safetypress_custom_login_url' => 'Benutzerdefinierte Login-URL',
            'safetypress_enable_custom_login' => 'Login-URL aktiv',
            'safetypress_login_redirect_url' => 'Login-Redirect URL',
        );
        
        $label = isset($labels[$option]) ? $labels[$option] : $option;
        
        $checkbox_options = array(
            'safetypress_enable_firewall', 'safetypress_enable_login_protection',
            'safetypress_enable_hardening', 'safetypress_enable_activity_log',
            'safetypress_enable_woo_protection', 'safetypress_email_notifications',
            'safetypress_enable_2fa', 'safetypress_hide_wp_version', 
            'safetypress_disable_xmlrpc', 'safetypress_disable_file_editor',
            'safetypress_enable_custom_login', 'safetypress_scheduled_scan_email'
        );
        
        if (in_array($option, $checkbox_options)) {
            $status = $new_value ? 'aktiviert' : 'deaktiviert';
            $message = sprintf('%s wurde %s', $label, $status);
        } else {
            $message = sprintf('%s geändert: %s → %s', $label, $old_value, $new_value);
        }
        
        safetypress()->logger->log('settings_changed', $message, array(
            'option' => $option,
            'old_value' => $old_value,
            'new_value' => $new_value,
        ));
    }
}
